package com.amity.socialcloud.sdk.core.data.invitation

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import co.amity.rxbridge.toRx3
import com.amity.socialcloud.sdk.core.data.reaction.paging.InvitationMediator
import com.amity.socialcloud.sdk.model.core.invitation.AmityInvitation
import com.amity.socialcloud.sdk.model.core.invitation.AmityInvitationStatus
import com.amity.socialcloud.sdk.model.core.invitation.AmityInvitationType
import com.ekoapp.ekosdk.EkoObjectRepository
import com.ekoapp.ekosdk.internal.api.socket.request.CreateInvitationsRequest
import com.ekoapp.ekosdk.internal.paging.DynamicQueryStreamPagerCreator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers

internal class InvitationRepository : EkoObjectRepository() {

    private fun getDefaultPageSize(): Int {
        return DEFAULT_PAGE_SIZE
    }

    @OptIn(ExperimentalPagingApi::class)
    fun getInvitationPagingData(
        type: AmityInvitationType,
        target: InvitationMediator.Target
    ): Flowable<PagingData<AmityInvitation>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = false
            ),
            dynamicQueryStreamMediator = InvitationMediator(
                type = type,
                target = target,
            ),
            pagingSourceFactory = {
                InvitationLocalDataStore().getInvitationsPagingSource(
                    type = type,
                    target = target
                )
            },
            modelMapper = InvitationModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    fun getMyCommunityInvitation(communityId: String): Single<List<AmityInvitation>> {
        return InvitationRemoteDataStore().getMyInvitations(
            type = AmityInvitationType.COMMUNITY_MEMBER,
            target = InvitationMediator.Target.COMMUNITY(communityId),
        )
            .flatMap { dto ->
                InvitationQueryPersister().persist(dto)
                    .andThen(Single.just((InvitationEntityMapper().map(dto))))
            }
            .map { entities ->
                if (entities.isEmpty()) {
                    emptyList()
                } else {
                    entities
                        .subList(0, 1)
                        .map(InvitationModelMapper()::map)
                }
            }
    }

    fun createCommunityInvitations(
        communityId: String,
        userIds: List<String>,
    ): Completable {
        return InvitationRemoteDataStore().createInvitations(
            request = CreateInvitationsRequest(
                type = AmityInvitationType.COMMUNITY_MEMBER.value,
                targetId = communityId,
                targetType = "community",
                userIds = userIds
            )
        )
            .flatMapCompletable { dto ->
                InvitationQueryPersister().persist(dto)
            }
    }

    fun acceptInvitation(invitationId: String): Completable {
        return InvitationRemoteDataStore().acceptInvitations(
            invitationId = invitationId,
        )
            .doOnSuccess {
                InvitationLocalDataStore().updateStatus(
                    invitationId = invitationId,
                    status = AmityInvitationStatus.APPROVED
                )
            }
            .subscribeOn(Schedulers.io())
            .ignoreElement()
    }

    fun rejectInvitation(invitationId: String): Completable {
        return InvitationRemoteDataStore().rejectInvitations(
            invitationId = invitationId,
        )
            .doOnSuccess {
                InvitationLocalDataStore().updateStatus(
                    invitationId = invitationId,
                    status = AmityInvitationStatus.REJECTED
                )
            }
            .subscribeOn(Schedulers.io())
            .ignoreElement()
    }

    fun cancelInvitation(invitationId: String): Completable {
        return InvitationRemoteDataStore().cancelInvitations(
            invitationId = invitationId,
        )
            .doOnSuccess {
                InvitationLocalDataStore().updateStatus(
                    invitationId = invitationId,
                    status = AmityInvitationStatus.CANCELLED
                )
            }
            .subscribeOn(Schedulers.io())
            .ignoreElement()
    }

}