package com.amity.socialcloud.sdk.social.data.post

import androidx.paging.PagingSource
import com.amity.socialcloud.sdk.api.social.post.query.AmityCommunityFeedSortOption
import com.amity.socialcloud.sdk.api.social.post.query.AmityUserFeedSortOption
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityException
import com.amity.socialcloud.sdk.model.social.feed.AmityFeedType
import com.amity.socialcloud.sdk.model.social.post.AmityPost
import com.ekoapp.ekosdk.internal.PostEntity
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.schedulers.Schedulers

internal class PostLocalDataStore {

    fun hasInLocal(postId: String): Boolean {
        return UserDatabase.get().postDao().getByIdNow(postId) != null
    }

    fun savePosts(posts: List<PostEntity>): Completable {
        return Completable.fromAction {
            UserDatabase.get().postDao().save(posts)
            UserDatabase.get().pinDao().updatePin()
        }
    }

    fun updateFeedType(postId: String, feedType: AmityFeedType) {
        return UserDatabase.get().postDao().updateFeedType(postId, feedType.apiKey)
    }

    fun getFeedType(postId: String): AmityFeedType {
        var feedType = AmityFeedType.NONE
        Completable.fromCallable {
            val post = UserDatabase.get().postDao().getByIdNow(postId)
            if (post != null) {
                feedType = AmityFeedType.enumOf(post.feedType)
            }
        }.subscribeOn(Schedulers.io())
            .blockingAwait()

        return feedType
    }

    fun observe(postId: String): Flowable<PostEntity> {
        return UserDatabase.get().postDao().observePost(postId)
            .map {
                if (it.isEmpty()) {
                    throw AmityException.create(
                        message = "item not found",
                        cause = null,
                        error = AmityError.ITEM_NOT_FOUND
                    )
                } else {
                    return@map it[0]
                }
            }
    }

    fun hardDelete(postId: String): Completable {
        return UserDatabase.get().postDao().deleteById(postId)
            .andThen(
                Completable.fromAction {
                    UserDatabase.get().pinDao().updatePin()
                }
            )
    }

    fun softDelete(postId: String): Completable {
        return UserDatabase.get().postDao().softDeleteById(postId)
            .andThen (
                Completable.fromAction {
                    UserDatabase.get().pinDao().updatePin()
                }
            )
    }

    fun invalidateChildPosts(postId: String, childPostIds: List<String>): Completable {
        return UserDatabase.get().postDao().invalidateChildPosts(postId, childPostIds)
    }

    fun getPost(postId: String): PostEntity? {
        var post: PostEntity? = null
        Completable.fromCallable {
            val entity = UserDatabase.get().postDao().getByIdNow(postId)
            if (entity != null) {
                post = entity
            }
        }.subscribeOn(Schedulers.io())
            .blockingAwait()
        return post
    }

    fun getPosts(postIds: List<String>): List<PostEntity> {
        var posts = emptyList<PostEntity>()
        Completable.fromCallable {
            val postDao = UserDatabase.get().postDao()
            val entities = postDao.getByIdsNow(postIds)
            if (entities.isNotEmpty()) {
                posts = entities.let { children ->
                    val orderById = postIds.withIndex().associate { it.value to it.index }
                    val sortedPost = children.sortedBy { orderById[it.postId] }
                    sortedPost
                }
            }
        }.subscribeOn(Schedulers.io())
            .blockingAwait()
        return posts
    }

    fun getCommunityPostsPagingSource(
        communityId: String,
        sortOption: AmityCommunityFeedSortOption,
        feedType: AmityFeedType,
        isDeleted: Boolean?,
        postTypes: List<AmityPost.DataType> = listOf()
    ): PagingSource<Int, PostEntity> {
        return UserDatabase.get().communityPostPagingDao().getCommunityPostPagingSource(
            communityId,
            feedType,
            isDeleted,
            sortOption.apiKey,
            postTypes
        )
    }

    fun getUserPostsPagingSource(
        userId: String,
        sortOption: AmityUserFeedSortOption,
        isDeleted: Boolean?,
        postTypes: List<AmityPost.DataType> = listOf()
    ): PagingSource<Int, PostEntity> {
        return UserDatabase.get().userPostPagingDao().getUserPostPagingSource(
            userId,
            isDeleted,
            sortOption.apiKey,
            postTypes
        )
    }

    fun getUserFeedPagingSource(
        userId: String,
        sortOption: AmityUserFeedSortOption,
        isDeleted: Boolean?,
        feedSources: List<String>,
        matchingOnlyParentPosts: Boolean,
        postTypes: List<AmityPost.DataType>,
    ): PagingSource<Int, PostEntity> {
        return UserDatabase.get().userFeedPagingDao().getUserFeedPagingSource(
            userId = userId,
            isDeleted = isDeleted,
            sortOption = sortOption,
            feedSources = feedSources,
            postTypes = postTypes,
            matchingOnlyParentPosts = matchingOnlyParentPosts
        )
    }

    fun getSemanticSearchPostsPagingSource(
        query: String,
        targetId: String?,
        targetType: String?,
        matchingOnlyParentPosts: Boolean,
        postTypes: List<AmityPost.DataType>,
    ): PagingSource<Int, PostEntity> {
        return UserDatabase.get().semanticSearchPostPagingDao().getSearchPostPagingSource(
            query = query,
            targetType = targetType,
            targetId = targetId,
            matchingOnlyParentPosts = matchingOnlyParentPosts,
            postTypes = postTypes
        )
    }

    fun searchPostsPagingSource(
        hashtags: List<String>,
        dataTypes: List<AmityPost.DataType>
    ): PagingSource<Int, PostEntity> {
        return UserDatabase.get().searchPostsPagingDao().searchPostsByHashtag(hashtags,dataTypes)
    }

    fun getGlobalPostsPagingSource(dataTypes: List<AmityPost.DataType>): PagingSource<Int, PostEntity> {
        return UserDatabase.get().globalFeedPagingDao().queryGlobalFeedPost(dataTypes)
    }

    fun getGlobalPostsV5PagingSource(dataTypes: List<AmityPost.DataType>): PagingSource<Int, PostEntity> {
        return UserDatabase.get().globalFeedV5PagingDao().queryGlobalFeedPost(dataTypes)
    }

    fun incrementCommentCount(postId: String) {
        UserDatabase.get().postDao().incrementCommentCount(postId)
    }

    fun decrementCommentCount(postId: String) {
        UserDatabase.get().postDao().decrementCommentCount(postId)
    }

    fun notifyPost(postId: String) {
        UserDatabase.get().postDao().updatePost(postId)
        UserDatabase.get().pinDao().updatePin()
    }

    fun getLatestPosts(
        targetId: String,
        targetType: String,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int,
        isDeleted: Boolean?,
        dataTypes: List<String>, feedType: AmityFeedType?
    ): Flowable<PostEntity> {
        return UserDatabase.get().postDao()
            .getLatestPost(
                targetId, targetType, isDeleted, dataTypes,
                dynamicQueryStreamKeyCreator.toMap().hashCode(), nonce, feedType
            )
    }

    fun getPostByIds(postIds: List<String>) : Flowable<List<PostEntity>> {
        return UserDatabase.get().postDao().getByPostIds(postIds)
    }

}
