package com.amity.socialcloud.sdk.chat.data.channel

import androidx.paging.PagingSource
import com.amity.socialcloud.sdk.core.data.session.SessionLocalDataStore
import com.amity.socialcloud.sdk.model.chat.channel.AmityChannel
import com.amity.socialcloud.sdk.model.chat.channel.AmityChannelFilter
import com.amity.socialcloud.sdk.model.core.tag.AmityTags
import com.ekoapp.ekosdk.EkoChannelWithMembershipAndExtra
import com.ekoapp.ekosdk.internal.EkoChannelEntity
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.schedulers.Schedulers
import org.joda.time.DateTime

internal class ChannelLocalDataStore {

    fun saveChannels(
        channels: List<EkoChannelEntity>,
    ): Completable {
        return Completable.fromAction {
            UserDatabase.get().channelDao().save(channels)
        }
    }

    fun getChannel(channelId: String): EkoChannelEntity? {
        return UserDatabase.get().channelDao().getByIdNow(channelId)
    }

    fun getChannelByInternalId(internalChannelId: String): EkoChannelEntity? {
        return UserDatabase.get().channelDao().getByInternalIdNow(internalChannelId)
    }

    fun observeChannel(channelId: String): Flowable<EkoChannelEntity> {
        return UserDatabase.get().channelDao().getChannel(channelId)
    }
    
    fun getMessagePreviewId(channelId: String): String? {
        return UserDatabase.get().channelDao().getMessagePreviewId(channelId)
    }

    fun getAllJoinedChannels(): Flowable<List<EkoChannelWithMembershipAndExtra>> {
        return UserDatabase.get().channelDao()
            .getAllJoinedChannels(SessionLocalDataStore().getActiveUserId())
    }

    fun updateDefaultSubChannelMessageCount(
        channelId: String,
        subChannelId: String,
        segment: Int
    ): Completable {
        return Completable.fromAction {
            val cachedChannel = getChannelByInternalId(subChannelId)
            if (cachedChannel != null) {
                UserDatabase.get().channelDao().updateMessageCount(channelId, segment)
            }
        }.subscribeOn(Schedulers.io())
    }

    fun updateLastActivity(channelId: String): Completable {
        return Completable.fromAction {
            UserDatabase.get().channelDao().updateLastActivity(channelId, DateTime.now())
        }.subscribeOn(Schedulers.io())
    }

    fun getChannelPagingSource(
        types: Set<AmityChannel.Type>,
        filter: AmityChannelFilter,
        includingTags: AmityTags,
        excludingTags: AmityTags,
        isDeleted: Boolean?,
    ): PagingSource<Int, EkoChannelEntity> {
        return UserDatabase.get().channelPagingDao().getChannelPagingSource(
            types,
            filter,
            includingTags,
            excludingTags,
            isDeleted
        )
    }
    
    fun getChannels(channelIds: List<String>): Flowable<List<EkoChannelEntity>> {
        return UserDatabase.get().channelListPagingDao().getChannels(channelIds)
    }

    fun getLatestChannel(
        types: Set<AmityChannel.Type>,
        filter: AmityChannelFilter,
        includingTags: AmityTags,
        excludingTags: AmityTags,
        isDeleted: Boolean?,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int
    ): Flowable<EkoChannelEntity> {
        return UserDatabase.get().channelDao().getLatestChannel(
            ChannelHelper.getChannelTypes(types).toTypedArray(),
            SessionLocalDataStore().getActiveUserId(),
            filter.memberships.toTypedArray(),
            includingTags.toTypedArray(),
            excludingTags.toTypedArray(),
            isDeleted,
            dynamicQueryStreamKeyCreator.toMap().hashCode(),
            nonce,
            DateTime.now()
        )
    }

    fun hasInLocal(channelId: String): Boolean {
        var hasInLocal = false
        Completable.fromCallable {
            val channelDao = UserDatabase.get().channelDao()
            val entity = channelDao.getByIdNow(channelId)
            if (entity != null) {
                hasInLocal = true
            }
        }.subscribeOn(Schedulers.io())
            .blockingAwait()
        return hasInLocal
    }
    
    fun updateMarkerHash(channelId: String, hash: Int) {
        UserDatabase.get().channelDao().updateMarkerHash(channelId, hash)
    }
    
    fun updateMessagePreview(channelId: String, messagePreviewId: String): Completable {
        return Completable.fromAction {
            UserDatabase.get().channelDao().updateMessagePreview(channelId, messagePreviewId)
        }.subscribeOn(Schedulers.io())
    }
    
    fun notifyChanges(channelId: String) {
        UserDatabase.get().channelDao().notifyChanges(channelId)
    }
    
    fun notifyChannelsChanges(channelIds: List<String>) {
        UserDatabase.get().channelDao().notifyChannelsChanges(channelIds, channelIds.hashCode())
    }

    fun hardDelete(channelId: String): Completable {
        return Completable.fromAction {
            UserDatabase.get().channelDao().deleteById(channelId)
        }.subscribeOn(Schedulers.io())
    }
}