package com.amity.socialcloud.sdk.social.data.community

import androidx.paging.PagingSource
import com.amity.socialcloud.sdk.api.social.community.query.AmityCommunitySortOption
import com.amity.socialcloud.sdk.model.core.tag.AmityTags
import com.amity.socialcloud.sdk.model.social.community.AmityCommunityFilter
import com.amity.socialcloud.sdk.model.social.community.AmityCommunityMembershipStatusFilter
import com.amity.socialcloud.sdk.model.social.feed.AmityFeedType
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.entity.CommunityEntity
import com.ekoapp.ekosdk.internal.entity.FeedEntity
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.schedulers.Schedulers
import org.joda.time.DateTime

internal class CommunityLocalDataStore {

    // To rename to "getCommunity"
    fun getCommunityById(communityId: String): CommunityEntity? {
        var community: CommunityEntity? = null
        Completable.fromCallable {
            val communityDao = UserDatabase.get().communityDao()
            val entity = communityDao.getByIdNow(communityId)
            if (entity != null) {
                community = entity
            }
        }.subscribeOn(Schedulers.io())
            .blockingAwait()
        return community
    }

    fun getCommunities(ids: List<String>): Flowable<List<CommunityEntity>> {
        return UserDatabase.get().communityDao().observeByCommunityIds(ids)
            .map { entities ->
                val orderById = ids.withIndex().associate { it.value to it.index }
                val sortedCommunities = entities.sortedBy { orderById[it.communityId] }
                sortedCommunities
            }
    }

    fun observeCommunity(communityId: String): Flowable<CommunityEntity> {
        return UserDatabase.get().communityDao().getByCommunityId(communityId)
    }

    fun saveCommunities(communities: List<CommunityEntity>): Completable {
        return Completable.fromAction { UserDatabase.get().communityDao().save(communities) }
    }

    fun deleteCommunity(communityId: String): Completable {
        return UserDatabase.get().communityDao().deleteByCommunityId(communityId)
    }

    fun getPostCount(targetId: String, feedType: AmityFeedType): Flowable<Int> {
        val feedDao = UserDatabase.get().feedDao()
        return feedDao.getCommunityFeed(targetId, feedType.apiKey).map(FeedEntity::postCount)
    }

    fun hasInLocal(communityId: String): Boolean {
        val communityDao = UserDatabase.get().communityDao()
        return communityDao.getByIdNow(communityId) != null
    }

    fun getCommunityPagingSource(
        keyword: String,
        categoryId: String,
        filter: AmityCommunityFilter,
        sortOption: AmityCommunitySortOption,
        isDeleted: Boolean?,
        includeDiscoverablePrivateCommunity: Boolean
    ): PagingSource<Int, CommunityEntity> {
        return UserDatabase.get().communityQueryPagingDao().getCommunityPagingSource(
            keyword = keyword,
            categoryId = categoryId,
            filter = filter,
            sortOption = sortOption,
            isDeleted = isDeleted,
            includeDiscoverablePrivateCommunity = includeDiscoverablePrivateCommunity,
        )
    }

    fun searchCommunityPagingSource(
        keyword: String,
        categoryId: String,
        filter: AmityCommunityFilter,
        sortOption: AmityCommunitySortOption,
        isDeleted: Boolean?,
        includeDiscoverablePrivateCommunity: Boolean
    ): PagingSource<Int, CommunityEntity> {
        return UserDatabase.get().communitySearchPagingDao().getCommunityPagingSource(
            keyword = keyword,
            categoryId = categoryId,
            filter = filter,
            sortOption = sortOption,
            isDeleted = isDeleted,
            includeDiscoverablePrivateCommunity = includeDiscoverablePrivateCommunity,
        )
    }

    fun getLatestCommunity(
        categoryId: String,
        filter: AmityCommunityFilter,
        isDeleted: Boolean?,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int
    ): Flowable<CommunityEntity> {
        return UserDatabase.get().communityDao().getLatestCommunity(
            categoryId,
            filter,
            isDeleted,
            dynamicQueryStreamKeyCreator.toMap().hashCode(),
            nonce,
            DateTime.now()
        )
    }

    fun getSearchCommunitiesPagingSource(
        query: String,
        filter: AmityCommunityMembershipStatusFilter?,
        tags: AmityTags?,
        categoryIds: List<String>?,
        //includeDiscoverablePrivateCommunity: Boolean? = null
    ): PagingSource<Int, CommunityEntity> {
        return UserDatabase.get().searchCommunitiesPagingDao().getSearchCommunitiesPagingSource(
            query = query,
            filter = filter,
            tags = tags,
            categoryIds = categoryIds,
            //includeDiscoverablePrivateCommunity = includeDiscoverablePrivateCommunity,
        )
    }

}