package com.ekoapp.ekosdk.internal.data.model

import androidx.room.Entity
import androidx.room.ForeignKey
import com.amity.socialcloud.sdk.social.data.post.PostRepository
import com.ekoapp.ekosdk.EkoFlag
import com.ekoapp.ekosdk.EkoObject
import com.ekoapp.ekosdk.internal.EkoLocalFlag
import com.ekoapp.ekosdk.internal.PostEntity
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.google.common.base.Objects
import com.google.gson.JsonObject
import com.google.gson.annotations.SerializedName
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.schedulers.Schedulers

@Entity(tableName = "post_flag",
		primaryKeys = ["postId"],
		foreignKeys = [
			ForeignKey(
				entity = PostEntity::class,
				parentColumns = ["postId"],
				childColumns = ["postId"],
				onDelete = ForeignKey.CASCADE
			)])
class EkoPostFlag constructor(
	val postId: String,
	@SerializedName("hashFlag")
	var flag: EkoFlag? = null,
	var localFlag: EkoLocalFlag? = null
) : EkoObject() {
	
	override fun getId(): String {
		return postId
	}
	
	val isFlaggedByMe: Boolean
		get() {
			flag?.let { flag ->
				val mightFlaggedByMe = flag.mightFlaggedByMe()
				val local = localFlag ?: EkoLocalFlag().apply {
					localFlag = this
				}
				if (mightFlaggedByMe && !Objects.equal(flag.hash, local.syncedUpHash)) {
					Single.just(postId)
							.filter { id: String -> flag.expired(id) }
							.flatMapSingle { postId: String -> PostRepository().isFlaggedByMe(postId) }
							.doOnSuccess { response: JsonObject ->
								local.isFlaggedByMe = response["result"].asBoolean
								local.syncedUpHash = flag.hash
								localFlag = local
								UserDatabase.get().postFlagDao().update(this)
							}
							.subscribe()
					return true
				}
				return mightFlaggedByMe && local.isFlaggedByMe
			} ?: localFlag?.let {
				Completable.fromAction {
					localFlag = null
					UserDatabase.get().postFlagDao().update(this)
				}.subscribeOn(Schedulers.io()).subscribe()
			}
			return false
		}
	
	override fun equals(obj: Any?): Boolean {
		return (super.equals(obj)
				&& Objects.equal(flag, (obj as EkoPostFlag).flag)
				&& Objects.equal(localFlag, obj.localFlag))
	}
	
	override fun hashCode(): Int {
		return Objects.hashCode(super.hashCode(), flag, localFlag)
	}
}