package com.ekoapp.ekosdk.internal.data.dao;

import androidx.annotation.NonNull;
import androidx.paging.DataSource;
import androidx.room.Dao;
import androidx.room.Query;
import androidx.room.Transaction;

import com.amity.socialcloud.sdk.api.social.community.query.AmityCommunitySortOption;
import com.amity.socialcloud.sdk.model.social.community.AmityCommunityFilter;
import com.ekoapp.ekosdk.internal.data.UserDatabase;
import com.ekoapp.ekosdk.internal.data.model.EkoCategoryEntity;
import com.ekoapp.ekosdk.internal.entity.CommunityEntity;
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator;

import org.joda.time.DateTime;

import java.util.List;

import io.reactivex.rxjava3.core.Completable;
import io.reactivex.rxjava3.core.Flowable;


@Dao
public abstract class EkoCommunityDao extends EkoObjectDao<CommunityEntity> {

    private final EkoCommunityWithCategoryDao communityWithCategoryDao = UserDatabase.get().communityWithCategoryDao();

    @Query("DELETE from community")
    public abstract void deleteAll();

    @Query("UPDATE community set isDeleted = 1 where communityId = :communityId")
    abstract Completable deleteByCommunityIdImpl(String communityId);

    public Completable deleteByCommunityId(String communityId) {
        return deleteByCommunityIdImpl(communityId);
    }

    @Query("SELECT *" +
            " from community" +
            " where community.communityId = :communityId" +
            " LIMIT 1")
    abstract Flowable<CommunityEntity> getByCommunityIdImpl(String communityId);

    public Flowable<CommunityEntity> getByCommunityId(String communityId) {
        return getByCommunityIdImpl(communityId);
    }

    @Query("SELECT *" +
            " from community" +
            " where community.communityId in (:communityIds)")
    abstract Flowable<List<CommunityEntity>> observeByCommunityIdsImpl(List<String> communityIds);

    public Flowable<List<CommunityEntity>> observeByCommunityIds(List<String> communityIds) {
        return observeByCommunityIdsImpl(communityIds);
    }

    @Transaction
    @Override
    public void insert(CommunityEntity community) {
        super.insert(community);
        EkoCategoryDao.update(community, communityWithCategoryDao, EkoCategoryEntity::create);
    }


    @Transaction
    @Override
    public void insert(@NonNull List<? extends CommunityEntity> communities) {
        super.insert(communities);
        EkoCategoryDao.update(communities, communityWithCategoryDao, EkoCategoryEntity::create);
    }

    @Transaction
    @Override
    public void update(CommunityEntity community) {
        super.update(community);
        EkoCategoryDao.update(community, communityWithCategoryDao, EkoCategoryEntity::create);
    }

    @Transaction
    @Override
    public void update(@NonNull List<? extends CommunityEntity> communities) {
        super.update(communities);
        EkoCategoryDao.update(communities, communityWithCategoryDao, EkoCategoryEntity::create);
    }

    @Query("SELECT * from community where communityId = :communityId LIMIT 1")
    abstract CommunityEntity getByIdNowImpl(String communityId);

    public CommunityEntity getByIdNow(String communityId) {
        return getByIdNowImpl(communityId);
    }

    @Query("SELECT *" +
            " from community" +
            " where community.communityId IN (:ids)")
    abstract List<CommunityEntity> getByIdsNowImpl(List<String> ids);

    public List<CommunityEntity> getByIdNow(List<String> ids) {
        return getByIdsNowImpl(ids);
    }

    @Transaction
    public void updateIsJoined(String communityId) {
        updateIsJoinedImpl(communityId);
    }

    @Query("UPDATE community set communityId = :communityId where communityId = :communityId")
    abstract void updateIsJoinedImpl(String communityId);

    @Query("UPDATE community set isJoined = :isJoined where communityId = :communityId")
    abstract void updateIsJoinedByCommunityIdImpl(String communityId, boolean isJoined);

    public void updateIsJoinedByCommunityId(String communityId, boolean isJoined) {
        updateIsJoinedByCommunityIdImpl(communityId, isJoined);
    }

    @Query("SELECT *" +
            " from community" +
            " where community.isJoined = (case when :isFilterByMembership then :isJoined else isJoined end)" +
            " and case when :isFilterByCategoryId then community.communityId in (SELECT communityId from category where categoryId in (:categoryId))" +
            " else community.communityId is not null end" +  // always true
            " and community.isDeleted = (case when :isDeleted is null then isDeleted else :isDeleted end)" +
            " and community.updatedAt > :now" +
            " and community.communityId not in " +
            "(" +
            "SELECT amity_paging_id.id" +
            " from amity_paging_id" +
            " where amity_paging_id.hash = (:hash)" +
            " and amity_paging_id.nonce = (:nonce) " +
            ")" +
            " order by community.updatedAt  desc" +
            " limit 1")
    abstract Flowable<CommunityEntity> getLatestCommunityImpl(Boolean isFilterByCategoryId,
                                                              String categoryId,
                                                              Boolean isFilterByMembership,
                                                              Boolean isJoined,
                                                              Boolean isDeleted,
                                                              int hash,
                                                              int nonce,
                                                              DateTime now);

    public Flowable<CommunityEntity> getLatestCommunity(String categoryId,
                                                        AmityCommunityFilter membershipFilter,
                                                        Boolean isDeleted,
                                                        int hash,
                                                        int nonce,
                                                        DateTime now) {
        return getLatestCommunityImpl(!categoryId.isEmpty(),
                categoryId,
                (membershipFilter != AmityCommunityFilter.ALL) ,
                (membershipFilter == AmityCommunityFilter.MEMBER),
                isDeleted,
                hash,
                nonce,
                now
        );
    }

    @Query("UPDATE community set communityId = :communityId where communityId = :communityId")
    public abstract void notifyChanges(String communityId);

}