package com.amity.socialcloud.sdk.social.data.event

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import co.amity.rxbridge.toRx3
import com.amity.socialcloud.sdk.common.AmityObjectRepository
import com.amity.socialcloud.sdk.common.ModelMapper
import com.amity.socialcloud.sdk.entity.social.event.EventEntity
import com.amity.socialcloud.sdk.model.core.tag.AmityTags
import com.amity.socialcloud.sdk.model.social.event.AmityEvent
import com.amity.socialcloud.sdk.model.social.event.AmityEventOriginType
import com.amity.socialcloud.sdk.model.social.event.AmityEventStatus
import com.amity.socialcloud.sdk.model.social.event.AmityEventType
import com.amity.socialcloud.sdk.social.data.event.paging.EventQueryMediator
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import com.ekoapp.ekosdk.internal.paging.QueryStreamPagerCreator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import org.joda.time.DateTime

internal class EventRepository : AmityObjectRepository<EventEntity, AmityEvent>() {

    override fun fetchAndSave(objectId: String): Completable {
        return EventRemoteDataStore().getEvent(objectId)
            .flatMapCompletable {
                EventQueryPersister().persist(it)
            }
    }

    override fun queryFromCache(objectId: String): EventEntity? {
        return EventLocalDataStore().getEvent(objectId)
    }

    override fun mapper(): ModelMapper<EventEntity, AmityEvent> {
        return EventModelMapper()
    }

    override fun observeFromCache(objectId: String): Flowable<EventEntity> {
        return EventLocalDataStore().observe(objectId)
    }

    fun getEvent(eventId: String): AmityEvent? {
        return EventLocalDataStore().getEvent(eventId)?.let {
            EventModelMapper().map(it)
        }
    }

    @OptIn(ExperimentalPagingApi::class)
    fun getEventPagingData(
        originType: AmityEventOriginType?,
        originId: String?,
        userId: String?,
        status: AmityEventStatus?,
        type: AmityEventType?,
        onlyAttendee: Boolean?,
        sortBy: com.amity.socialcloud.sdk.api.social.event.query.AmityEventSortOption,
        orderBy: com.amity.socialcloud.sdk.api.social.event.query.AmityEventOrderOption
    ): Flowable<PagingData<AmityEvent>> {
        val pagerCreator = QueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = 20,
                enablePlaceholders = true,
                initialLoadSize = 10,
                prefetchDistance = 0
            ),
            queryStreamMediator = EventQueryMediator(
                originType = originType,
                originId = originId,
                userId = userId,
                status = status,
                type = type,
                onlyAttendee = onlyAttendee,
                sortBy = sortBy,
                orderBy = orderBy
            ),
            pagingSourceFactory = {
                EventLocalDataStore().getEventPagingSource(
                    originType = originType,
                    originId = originId,
                    userId = userId,
                    status = status,
                    type = type,
                    onlyAttendee = onlyAttendee,
                    sortBy = sortBy,
                    orderBy = orderBy
                )
            },
            modelMapper = EventModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    fun createEvent(
        title: String?,
        description: String?,
        type: AmityEventType?,
        isInviteOnly: Boolean?,
        startTime: DateTime?,
        endTime: DateTime, // Made non-nullable (required)
        originType: AmityEventOriginType,
        originId: String,
        location: String?,
        externalUrl: String?,
        coverImageFileId: String?,
        tags: AmityTags?,
        timezone: String?
    ): Single<AmityEvent> {
        return EventRemoteDataStore().createEvent(
            title = title,
            description = description,
            type = type,
            isInviteOnly = isInviteOnly,
            startTime = startTime,
            endTime = endTime,
            originType = originType,
            originId = originId,
            location = location,
            externalUrl = externalUrl,
            coverImageFileId = coverImageFileId,
            tags = tags,
            timezone = timezone
        ).flatMap { dto ->
            EventQueryPersister().persist(dto)
                .andThen(
                    dto.events?.firstOrNull()?.eventId?.let { eventId ->
                        // For now, directly map from DTO to model as we don't have local caching
                        val entity = dto.events?.firstOrNull()?.let { EventEntityMapper().map(listOf(it)).firstOrNull() }
                        entity?.let { Single.just(EventModelMapper().map(it)) }
                            ?: Single.error(Exception("Event not found in response"))
                    } ?: Single.error(Exception("No events in response"))
                )
        }
    }

    fun updateEvent(
        eventId: String,
        title: String?,
        description: String?,
        type: AmityEventType?,
        isInviteOnly: Boolean?,
        startTime: DateTime?,
        endTime: DateTime?,
        location: String?,
        externalUrl: String?,
        coverImageFileId: String?,
        tags: AmityTags?,
        timezone: String?
    ): Single<AmityEvent> {
        return EventRemoteDataStore().updateEvent(
            eventId = eventId,
            title = title,
            description = description,
            type = type,
            isInviteOnly = isInviteOnly,
            startTime = startTime,
            endTime = endTime,
            location = location,
            externalUrl = externalUrl,
            coverImageFileId = coverImageFileId,
            tags = tags,
            timezone = timezone
        ).flatMap { dto ->
            EventQueryPersister().persist(dto)
                .andThen(
                    dto.events?.firstOrNull()?.eventId?.let { eventId ->
                        // For now, directly map from DTO to model as we don't have local caching
                        val entity = dto.events?.firstOrNull()?.let { EventEntityMapper().map(listOf(it)).firstOrNull() }
                        entity?.let { Single.just(EventModelMapper().map(it)) }
                            ?: Single.error(Exception("Event not found in response"))
                    } ?: Single.error(Exception("No events in response"))
                )
        }
    }

    fun deleteEvent(eventId: String): Completable {
        return EventRemoteDataStore().deleteEvent(eventId)
            .flatMapCompletable {
                when (it.isSuccess) {
                    true -> {
                        Completable.fromAction {
                            val eventDao = com.ekoapp.ekosdk.internal.data.UserDatabase.get().eventDao()
                            eventDao.deleteById(eventId)
                        }
                    }
                    else -> Completable.complete()
                }
            }
    }

    fun getLatestEvent(
        originType: AmityEventOriginType?,
        originId: String?,
        userId: String?,
        status: AmityEventStatus?,
        type: AmityEventType?,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int,
        sortBy: com.amity.socialcloud.sdk.api.social.event.query.AmityEventSortOption,
        orderBy: com.amity.socialcloud.sdk.api.social.event.query.AmityEventOrderOption
    ): Flowable<AmityEvent> {
        return EventLocalDataStore().getLatestEvent(
            originType = originType,
            originId = originId,
            userId = userId,
            status = status,
            type = type,
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            nonce = nonce,
            sortBy = sortBy,
            orderBy = orderBy
        ).map { EventModelMapper().map(it) }
    }

    internal fun notifyChanges(eventId: String) {
        EventLocalDataStore().notifyChanges(eventId)
    }
}
