package com.amity.socialcloud.sdk.chat.data.subchannel.paging

import androidx.paging.ExperimentalPagingApi
import co.amity.rxbridge.toRx2
import com.amity.socialcloud.sdk.chat.data.marker.subchannel.SubChannelMarkerRepository
import com.amity.socialcloud.sdk.chat.data.subchannel.SubChannelQueryPersister
import com.amity.socialcloud.sdk.chat.data.subchannel.SubChannelRemoteDataStore
import com.amity.socialcloud.sdk.chat.data.subchannel.SubChannelRepository
import com.amity.socialcloud.sdk.core.MarkerSyncEngine
import com.amity.socialcloud.sdk.infra.retrofit.request.QueryOptionsRequestParams
import com.amity.socialcloud.sdk.model.chat.subchannel.AmitySubChannel
import com.ekoapp.ekosdk.internal.SubChannelEntity
import com.ekoapp.ekosdk.internal.api.dto.SubChannelQueryDto
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.mediator.DynamicQueryStreamMediator
import com.ekoapp.ekosdk.internal.token.DynamicQueryStreamQueryToken
import com.ekoapp.ekosdk.internal.token.QueryStreamQueryToken
import io.reactivex.Completable
import io.reactivex.Flowable
import io.reactivex.Single

@OptIn(ExperimentalPagingApi::class)
class SubChannelMediator(
    private val channelId: String,
    private val excludeMainSubChannel: Boolean,
    private val isDeleted: Boolean
) : DynamicQueryStreamMediator<SubChannelEntity, SubChannelQueryDto, AmitySubChannel>(
    nonce = AmityNonce.SUB_CHANNEL_LIST,
    dynamicQueryStreamKeyCreator = SubChannelKeyCreator(
        channelId = channelId,
        isDeleted = isDeleted
    )
) {

    override fun forceRefresh() = true

    override fun provideReactorPublisher(): Flowable<AmitySubChannel> {
        return SubChannelRepository().getLatestSubChannel(
            isDeleted = isDeleted,
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            nonce = nonce
        ).toRx2()
    }

    override fun getFirstPageRequest(pageSize: Int): Single<SubChannelQueryDto> {
        return getRequest(limit = pageSize)
    }

    override fun getFetchByTokenRequest(token: String): Single<SubChannelQueryDto> {
        return getRequest(token = token)
    }

    override fun persistResponse(dto: SubChannelQueryDto): Completable {
        return dto.subChannels
            .filter { MarkerSyncEngine.isMarkerSyncSupport(it.channelType) }
            .mapNotNull { it.subChannelId }
            .let (SubChannelMarkerRepository()::fetchSubChannelMarker)
            .onErrorComplete()
            .andThen(SubChannelQueryPersister().persist(dto))
            .toRx2()
    }

    override fun convertResponseToQueryToken(dto: SubChannelQueryDto): QueryStreamQueryToken {
        return DynamicQueryStreamQueryToken(
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            next = dto.token?.next,
            previous = dto.token?.previous,
            primaryKeys = dto.subChannels.map { it.subChannelId ?: "" }
        )
    }

    private fun getRequest(
        limit: Int? = null,
        token: String? = null
    ): Single<SubChannelQueryDto> {
        return SubChannelRemoteDataStore().querySubChannels(
            internalChannelId = channelId,
            excludeMainSubChannel = excludeMainSubChannel,
            isDeleted = isDeleted,
            options = QueryOptionsRequestParams(limit, token)
        ).toRx2()
    }

}