package com.amity.socialcloud.sdk.api.social.event.update

import com.amity.socialcloud.sdk.model.core.tag.AmityTags
import com.amity.socialcloud.sdk.model.social.event.AmityEvent
import com.amity.socialcloud.sdk.model.social.event.AmityEventType
import com.amity.socialcloud.sdk.social.domain.event.update.EventUpdateUseCase
import io.reactivex.rxjava3.core.Single
import org.joda.time.DateTime

class AmityEventUpdateOptions private constructor(
    private val eventId: String,
    private val title: String?,
    private val description: String?,
    private val type: AmityEventType?,
    private val isInviteOnly: Boolean?,
    private val startTime: DateTime?,
    private val endTime: DateTime?,
    private val location: String?,
    private val externalUrl: String?,
    private val coverImageFileId: String?,
    private val tags: AmityTags?,
    private val timezone: String?
) {

    /**
     * Updates an event
     * @return Async AmityEvent as Single<AmityEvent>.
     */
    fun apply(): Single<AmityEvent> {
        return EventUpdateUseCase().execute(
            eventId = eventId,
            title = title,
            description = description,
            type = type,
            isInviteOnly = isInviteOnly,
            startTime = startTime,
            endTime = endTime,
            location = location,
            externalUrl = externalUrl,
            coverImageFileId = coverImageFileId,
            tags = tags,
            timezone = timezone
        )
    }

    class Builder internal constructor(private val eventId: String) {
        private var title: String? = null
        private var description: String? = null
        private var type: AmityEventType? = null
        private var isInviteOnly: Boolean? = null
        private var startTime: DateTime? = null
        private var endTime: DateTime? = null
        private var location: String? = null
        private var externalUrl: String? = null
        private var coverImageFileId: String? = null
        private var tags: AmityTags? = null
        private var timezone: String? = null

        /**
         * @param [title] to update the event title
         * @return the same instance of AmityEventUpdateOptions.Builder
         */
        fun setTitle(title: String): Builder {
            return apply {
                this.title = title
            }
        }

        /**
         * @param [title] to update the event title
         * @return the same instance of AmityEventUpdateOptions.Builder
         */
        fun title(title: String): Builder = setTitle(title)

        /**
         * @param [description] to update the event description
         * @return the same instance of AmityEventUpdateOptions.Builder
         */
        fun setDescription(description: String): Builder {
            return apply {
                this.description = description
            }
        }

        /**
         * @param [description] to update the event description
         * @return the same instance of AmityEventUpdateOptions.Builder
         */
        fun description(description: String): Builder = setDescription(description)

        /**
         * @param [type] to update the event type
         * @return the same instance of AmityEventUpdateOptions.Builder
         */
        fun setType(type: AmityEventType): Builder {
            return apply {
                this.type = type
            }
        }

        /**
         * @param [type] to update the event type
         * @return the same instance of AmityEventUpdateOptions.Builder
         */
        fun type(type: AmityEventType): Builder = setType(type)

        /**
         * @param [isInviteOnly] to update whether the event is invite-only
         * @return the same instance of AmityEventUpdateOptions.Builder
         */
        fun setIsInviteOnly(isInviteOnly: Boolean): Builder {
            return apply {
                this.isInviteOnly = isInviteOnly
            }
        }

        /**
         * @param [isInviteOnly] to update whether the event is invite-only
         * @return the same instance of AmityEventUpdateOptions.Builder
         */
        fun isInviteOnly(isInviteOnly: Boolean): Builder = setIsInviteOnly(isInviteOnly)

        /**
         * @param [startTime] to update the event start time
         * @return the same instance of AmityEventUpdateOptions.Builder
         */
        fun setStartTime(startTime: DateTime): Builder {
            return apply {
                this.startTime = startTime
            }
        }

        /**
         * @param [startTime] to update the event start time
         * @return the same instance of AmityEventUpdateOptions.Builder
         */
        fun startTime(startTime: DateTime): Builder = setStartTime(startTime)

        /**
         * @param [endTime] to update the event end time
         * @return the same instance of AmityEventUpdateOptions.Builder
         */
        fun setEndTime(endTime: DateTime): Builder {
            return apply {
                this.endTime = endTime
            }
        }

        /**
         * @param [endTime] to update the event end time
         * @return the same instance of AmityEventUpdateOptions.Builder
         */
        fun endTime(endTime: DateTime): Builder = setEndTime(endTime)

        /**
         * @param [location] to update the event location
         * @return the same instance of AmityEventUpdateOptions.Builder
         */
        fun setLocation(location: String): Builder {
            return apply {
                this.location = location
            }
        }

        /**
         * @param [location] to update the event location
         * @return the same instance of AmityEventUpdateOptions.Builder
         */
        fun location(location: String): Builder = setLocation(location)

        /**
         * @param [externalUrl] to update the event external URL
         * @return the same instance of AmityEventUpdateOptions.Builder
         */
        fun setExternalUrl(externalUrl: String): Builder {
            return apply {
                this.externalUrl = externalUrl
            }
        }

        /**
         * @param [externalUrl] to update the event external URL
         * @return the same instance of AmityEventUpdateOptions.Builder
         */
        fun externalUrl(externalUrl: String): Builder = setExternalUrl(externalUrl)

        /**
         * @param [coverImageFileId] to update the event cover image file ID
         * @return the same instance of AmityEventUpdateOptions.Builder
         */
        fun setCoverImageFileId(coverImageFileId: String): Builder {
            return apply {
                this.coverImageFileId = coverImageFileId
            }
        }

        /**
         * @param [coverImageFileId] to update the event cover image file ID
         * @return the same instance of AmityEventUpdateOptions.Builder
         */
        fun coverImageFileId(coverImageFileId: String): Builder = setCoverImageFileId(coverImageFileId)

        /**
         * @param [tags] to update the event tags
         * @return the same instance of AmityEventUpdateOptions.Builder
         */
        fun setTags(tags: AmityTags): Builder {
            return apply {
                this.tags = tags
            }
        }

        /**
         * @param [tags] to update the event tags
         * @return the same instance of AmityEventUpdateOptions.Builder
         */
        fun tags(tags: AmityTags): Builder = setTags(tags)

        /**
         * @param [tags] to update the event tags from a list of strings
         * @return the same instance of AmityEventUpdateOptions.Builder
         */
        fun setTags(tags: List<String>): Builder {
            return apply {
                this.tags = AmityTags(tags)
            }
        }

        /**
         * @param [tags] to update the event tags from a list of strings
         * @return the same instance of AmityEventUpdateOptions.Builder
         */
        fun tags(tags: List<String>): Builder = setTags(tags)

        /**
         * @param [timezone] to update the event timezone (e.g., "Asia/Bangkok")
         * @return the same instance of AmityEventUpdateOptions.Builder
         */
        fun setTimezone(timezone: String): Builder {
            return apply {
                this.timezone = timezone
            }
        }

        /**
         * @param [timezone] to update the event timezone (e.g., "Asia/Bangkok")
         * @return the same instance of AmityEventUpdateOptions.Builder
         */
        fun timezone(timezone: String): Builder = setTimezone(timezone)

        /**
         * Instantiates AmityEventUpdateOptions with built params
         * @return new instance of AmityEventUpdateOptions
         */
        fun build(): AmityEventUpdateOptions {
            return AmityEventUpdateOptions(
                eventId = eventId,
                title = title,
                description = description,
                type = type,
                isInviteOnly = isInviteOnly,
                startTime = startTime,
                endTime = endTime,
                location = location,
                externalUrl = externalUrl,
                coverImageFileId = coverImageFileId,
                tags = tags,
                timezone = timezone
            )
        }
    }
}
