package com.amity.socialcloud.sdk.social.data.event

import androidx.paging.PagingSource
import com.amity.socialcloud.sdk.entity.social.event.EventResponseEntity
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityException
import com.amity.socialcloud.sdk.model.social.event.AmityEventResponseStatus
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.schedulers.Schedulers

internal class EventResponseLocalDataStore {

    fun hasInLocal(rsvpId: String): Boolean {
        return UserDatabase.get().eventResponseDao().getByIdNow(rsvpId) != null
    }

    fun saveEventResponses(responses: List<EventResponseEntity>): Completable {
        return Completable.fromAction {
            UserDatabase.get().eventResponseDao().save(responses)
        }
    }

    fun observe(rsvpId: String): Flowable<EventResponseEntity> {
        return UserDatabase.get().eventResponseDao().observeEventResponse(rsvpId)
            .map {
                if (it.isEmpty()) {
                    throw AmityException.create(
                        message = "item not found",
                        cause = null,
                        error = AmityError.ITEM_NOT_FOUND
                    )
                } else {
                    return@map it[0]
                }
            }
    }

    fun getEventResponse(rsvpId: String): EventResponseEntity? {
        var response: EventResponseEntity? = null
        Completable.fromCallable {
            val entity = UserDatabase.get().eventResponseDao().getByIdNow(rsvpId)
            if (entity != null) {
                response = entity
            }
        }.subscribeOn(Schedulers.io())
            .blockingAwait()
        return response
    }

    fun getEventResponses(rsvpIds: List<String>): List<EventResponseEntity> {
        var responses = emptyList<EventResponseEntity>()
        Completable.fromCallable {
            val eventResponseDao = UserDatabase.get().eventResponseDao()
            val entities = eventResponseDao.getByIdsNow(rsvpIds)
            if (entities.isNotEmpty()) {
                responses = entities.let { results ->
                    val orderById = rsvpIds.withIndex().associate { it.value to it.index }
                    val sortedResponses = results.sortedBy { orderById[it.rsvpId] }
                    sortedResponses
                }
            }
        }.subscribeOn(Schedulers.io())
            .blockingAwait()
        return responses
    }

    fun getEventResponsePagingSource(
        eventId: String,
        status: AmityEventResponseStatus?
    ): PagingSource<Int, EventResponseEntity> {
        return UserDatabase.get().eventResponsePagingDao().getEventResponsePagingSource(
            eventId = eventId,
            status = status
        )
    }

    fun getLatestEventResponse(
        eventId: String,
        status: AmityEventResponseStatus?,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int
    ): Flowable<EventResponseEntity> {
        return UserDatabase.get().eventResponseDao().getLatestEventResponse(
            eventId = eventId,
            status = status?.apiKey,
            hash = dynamicQueryStreamKeyCreator.toMap().hashCode(),
            nonce = nonce,
            now = org.joda.time.DateTime.now()
        )
    }

    fun notifyChanges(rsvpId: String) {
        UserDatabase.get().eventResponseDao().notifyChanges(rsvpId)
    }
}
