package com.amity.socialcloud.sdk.social.data.post.paging

import androidx.paging.ExperimentalPagingApi
import co.amity.rxbridge.toRx2
import com.amity.socialcloud.sdk.model.social.feed.AmityFeedType
import com.amity.socialcloud.sdk.model.social.post.AmityPost
import com.amity.socialcloud.sdk.social.data.post.PostQueryPersister
import com.amity.socialcloud.sdk.social.data.post.PostRemoteDataStore
import com.ekoapp.ekosdk.internal.PostEntity
import com.ekoapp.ekosdk.internal.api.dto.EkoPostQueryDto
import com.ekoapp.ekosdk.internal.data.AmityNonce
import com.ekoapp.ekosdk.internal.mediator.QueryStreamMediator
import com.ekoapp.ekosdk.internal.token.QueryStreamQueryToken
import io.reactivex.Completable
import io.reactivex.Single

@ExperimentalPagingApi
internal class SearchPostMediator(
    private val targetType: String?,
    private val targetId: String?,
    private val hashtags: List<String>,
    private val feedType: AmityFeedType?,
    private val dataTypes: List<AmityPost.DataType>,
    private val includeMixedStructure: Boolean
) : QueryStreamMediator<PostEntity, EkoPostQueryDto>(
    nonce = AmityNonce.SEARCH_POST_LIST,
    queryStreamKeyCreator = SearchPostKeyCreator(
        targetType = targetType,
        targetId = targetId,
        hashtags = hashtags,
        feedType = feedType,
        dataTypes = dataTypes,
        includeMixedStructure = includeMixedStructure
    )
) {

    override fun forceRefresh() = true

    override fun getFirstPageRequest(pageSize: Int): Single<EkoPostQueryDto> {
        return getRequest(limit = pageSize)
    }

    override fun getFetchByTokenRequest(token: String): Single<EkoPostQueryDto> {
        return getRequest(token = token)
    }

    override fun persistResponse(dto: EkoPostQueryDto): Completable {
        return PostQueryPersister().persist(dto).toRx2()
    }

    override fun convertResponseToQueryToken(dto: EkoPostQueryDto): QueryStreamQueryToken {
        return QueryStreamQueryToken(
            queryStreamKeyCreator = queryStreamKeyCreator,
            next = dto.paging.next,
            previous = dto.paging.previous,
            primaryKeys = dto.posts.map { it.postId } ?: emptyList()
        )
    }

    private fun getRequest(
        limit: Int? = null,
        token: String? = null
    ): Single<EkoPostQueryDto> {
        val matchingOnlyParentPost = getDataTypes()?.isEmpty() ?: true
        return PostRemoteDataStore().searchPosts(
            targetType = targetType,
            targetId = targetId,
            hashtags = hashtags,
            isDeleted = false, // Only fetch non-deleted posts for search results
            feedType = feedType?.apiKey,
            dataTypes = getDataTypes(),
            matchingOnlyParentPost = matchingOnlyParentPost,
            includeMixedStructure = includeMixedStructure,
            limit = limit,
            token = token
        ).toRx2()
    }

    private fun getDataTypes(): List<String>? {
        if (dataTypes.isNotEmpty()) {
            val typeArrayList = arrayListOf<String>()
            dataTypes.forEach { typeArrayList.add(it.getApiKey()) }
            return typeArrayList
        }
        return null
    }
}