package com.amity.socialcloud.sdk.model.social.event

import android.os.Parcelable
import androidx.paging.PagingData
import com.amity.socialcloud.sdk.core.JsonObjectParceler
import com.amity.socialcloud.sdk.model.core.file.AmityImage
import com.amity.socialcloud.sdk.model.core.user.AmityUser
import com.amity.socialcloud.sdk.model.social.community.AmityCommunity
import com.amity.socialcloud.sdk.social.domain.event.EventCreateRSVPUseCase
import com.amity.socialcloud.sdk.social.domain.event.EventGetMyRSVPUseCase
import com.amity.socialcloud.sdk.social.domain.event.EventRSVPQueryUseCase
import com.amity.socialcloud.sdk.social.domain.event.EventUpdateRSVPUseCase
import com.ekoapp.ekosdk.ReactorObject
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import kotlinx.parcelize.Parcelize
import kotlinx.parcelize.TypeParceler
import org.amity.types.ObjectId
import org.joda.time.DateTime

@Parcelize
@TypeParceler<JsonObject?, JsonObjectParceler>
data class AmityEvent internal constructor(
    private val publicId: String = ObjectId.get().toHexString(),
    private val eventId: String = ObjectId.get().toHexString(),
    private val title: String,
    private val description: String,
    private val type: AmityEventType,
    private val status: AmityEventStatus,
    private val startTime: DateTime,
    private val endTime: DateTime?,
    private val coverImageFileId: String?,
    internal var coverImage: AmityImage?,
    private val originType: AmityEventOriginType,
    private val originId: String,
    private val userId: String,
    internal var creator: AmityUser?,
    private val discussionCommunityId: String?,
    internal var targetCommunity: AmityCommunity?,
    private val postId: String?,
    private val rsvpCount: Int,
    private val interestedCount: Int,
    private val isLocationVisible: Boolean,
    private val location: String?,
    private val externalUrl: String?,
    private val createdAt: DateTime,
    private val updatedAt: DateTime,
    internal val path: String
) : Parcelable, ReactorObject {

    fun getPublicId(): String {
        return publicId
    }

    fun getEventId(): String {
        return eventId
    }

    fun getTitle(): String {
        return title
    }

    fun getDescription(): String {
        return description
    }

    fun getType(): AmityEventType {
        return type
    }

    fun getStatus(): AmityEventStatus {
        return status
    }

    fun getStartTime(): DateTime {
        return startTime
    }

    fun getEndTime(): DateTime? {
        return endTime
    }

    fun getCoverImageFileId(): String? {
        return coverImageFileId
    }

    fun getCoverImage(): AmityImage? {
        return coverImage
    }

    fun getOriginType(): AmityEventOriginType {
        return originType
    }

    fun getOriginId(): String {
        return originId
    }

    fun getUserId(): String {
        return userId
    }

    fun getCreator(): AmityUser? {
        return creator
    }

    fun getDiscussionCommunityId(): String? {
        return discussionCommunityId
    }

    fun getTargetCommunity(): AmityCommunity? {
        return targetCommunity
    }

    fun getPostId(): String? {
        return postId
    }

    fun getRsvpCount(): Int {
        return rsvpCount
    }

    fun getInterestedCount(): Int {
        return interestedCount
    }

    fun isLocationVisible(): Boolean {
        return isLocationVisible
    }

    fun getLocation(): String? {
        return location
    }

    fun getExternalUrl(): String? {
        return externalUrl
    }

    fun getCreatedAt(): DateTime {
        return createdAt
    }

    fun getUpdatedAt(): DateTime {
        return updatedAt
    }

    override fun updatedAt(): DateTime {
        return updatedAt
    }

    override fun uniqueId(): String {
        return eventId
    }

    /**
     * Creates an RSVP for this event.
     * 
     * API: event.rsvp.create
     * API Endpoint: POST /api/v1/events/{event-id}/rsvp
     * 
     * @param status The RSVP status (GOING, INTERESTED, NOT_GOING)
     * @return Single<AmityEventResponse> The created RSVP response
     * 
     * @PublicAPI
     */
    fun createRSVP(status: AmityEventResponseStatus): Single<AmityEventResponse> {
        return EventCreateRSVPUseCase().createRSVP(eventId, status)
    }

    /**
     * Updates an existing RSVP for this event.
     * 
     * API: event.rsvp.update
     * API Endpoint: PUT /api/v1/events/{event-id}/rsvp
     * 
     * @param status The updated RSVP status (GOING, INTERESTED, NOT_GOING)
     * @return Single<AmityEventResponse> The updated RSVP response
     * 
     * @PublicAPI
     */
    fun updateRSVP(status: AmityEventResponseStatus): Single<AmityEventResponse> {
        return EventUpdateRSVPUseCase().updateRSVP(eventId, status)
    }

    /**
     * Queries RSVPs for this event with optional status filter.
     * Returns a live collection that automatically updates when RSVPs change.
     * 
     * API: event.rsvp.query
     * API Endpoint: GET /api/v1/events/{event-id}/rsvp
     * 
     * @param status Optional status filter (GOING, INTERESTED, NOT_GOING). Pass null to get all RSVPs.
     * @return Flowable<PagingData<AmityEventResponse>> Live collection of RSVPs
     * 
     * @PublicAPI
     */
    fun getRSVPs(status: AmityEventResponseStatus? = null): Flowable<PagingData<AmityEventResponse>> {
        return EventRSVPQueryUseCase().execute(
            eventId = eventId,
            status = status
        )
    }

    /**
     * Gets the current user's RSVP for this event.
     * 
     * API: event.rsvp.me
     * API Endpoint: GET /api/v1/events/{event-id}/me/rsvp
     * 
     * @return Single<AmityEventResponse> The current user's RSVP response
     * 
     * @PublicAPI
     */
    fun getMyRSVP(): Single<AmityEventResponse> {
        return EventGetMyRSVPUseCase().getMyRSVP(eventId)
    }
}
