package com.amity.socialcloud.sdk.core

import com.amity.socialcloud.sdk.api.core.AmityCoreClient
import com.amity.socialcloud.sdk.api.core.encryption.AmityDBEncryption
import com.amity.socialcloud.sdk.api.core.endpoint.AmityEndpoint
import com.amity.socialcloud.sdk.api.core.presence.AmityPresenceService
import com.amity.socialcloud.sdk.api.core.presence.AmityUserPresenceRepository
import com.amity.socialcloud.sdk.api.core.session.AmitySessionEstablisher
import com.amity.socialcloud.sdk.api.core.session.AmityVisitorEstablisher
import com.amity.socialcloud.sdk.chat.data.marker.user.ObserveUserMarker
import com.amity.socialcloud.sdk.chat.data.marker.user.UserUnreadModelMapper
import com.amity.socialcloud.sdk.core.engine.analytics.AnalyticsEngine
import com.amity.socialcloud.sdk.core.presence.PresenceSyncEngine
import com.amity.socialcloud.sdk.core.session.SessionError
import com.amity.socialcloud.sdk.core.session.SessionStateManager
import com.amity.socialcloud.sdk.core.session.component.DatabaseSessionComponent
import com.amity.socialcloud.sdk.core.session.component.TokenRenewalSessionComponent
import com.amity.socialcloud.sdk.core.session.component.TokenWatcherSessionComponent
import com.amity.socialcloud.sdk.core.session.component.UserSettingSessionComponent
import com.amity.socialcloud.sdk.core.session.eventbus.AppEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionLifeCycleEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionStateEventBus
import com.amity.socialcloud.sdk.core.session.model.AppEvent
import com.amity.socialcloud.sdk.core.session.model.SessionState
import com.amity.socialcloud.sdk.infra.mqtt.AmityMqttClient
import com.amity.socialcloud.sdk.log.AmityLog
import com.amity.socialcloud.sdk.model.chat.message.AmityMessageAttachment
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityException
import com.amity.socialcloud.sdk.model.core.file.AmityFileAccessType
import com.amity.socialcloud.sdk.model.core.reaction.AmityLiveReactionReferenceType
import com.amity.socialcloud.sdk.model.core.reaction.live.AmityLiveReaction
import com.amity.socialcloud.sdk.model.core.session.SessionHandler
import com.amity.socialcloud.sdk.model.core.unread.UserUnread
import com.ekoapp.ekosdk.internal.EkoMessageEntity
import com.ekoapp.ekosdk.internal.api.AmityHttpClient
import com.ekoapp.ekosdk.internal.api.http.AmityNetworkActivity
import com.ekoapp.ekosdk.internal.data.EkoDatabase
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.model.AmityUploadUrl
import com.ekoapp.ekosdk.internal.data.model.EkoAccount
import com.ekoapp.ekosdk.internal.data.model.EkoApiKey
import com.ekoapp.ekosdk.internal.data.model.EkoHttpUrl
import com.ekoapp.ekosdk.internal.data.model.EkoMqttUrl
import com.ekoapp.ekosdk.internal.init.AmityCoreSDKInitializer
import com.ekoapp.ekosdk.internal.init.EkoSdkInitProvider
import com.ekoapp.ekosdk.internal.util.AppContext
import com.google.common.base.Objects
import com.hivemq.client.mqtt.MqttClientState
import io.reactivex.rxjava3.core.BackpressureStrategy
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.CompletableEmitter
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Maybe
import io.reactivex.rxjava3.schedulers.Schedulers
import io.reactivex.rxjava3.subjects.CompletableSubject
import io.reactivex.rxjava3.subjects.PublishSubject
import org.joda.time.DateTime

internal object CoreClient {

    private const val TAG = "AmityCoreClient"

    internal var millisTimeDiff = 0

    internal val networkActivityPublisher = PublishSubject.create<AmityNetworkActivity>()
    internal fun getServerTime(): DateTime {
        return DateTime.now().plusMillis(millisTimeDiff)
    }

    private var mqttClient: AmityMqttClient? = null
    private var renewalManager: TokenRenewalSessionComponent? = null

    internal var presenceSyncEngine: PresenceSyncEngine? = null
    private var analyticsEngine: AnalyticsEngine? = null

    internal var sessionLifeCycleEventBus: SessionLifeCycleEventBus? = null
    internal var appEventBus: AppEventBus? = null
    internal var sessionStateEventBus: SessionStateEventBus? = null
    internal var sessionStateManager: SessionStateManager? = null
    val currentSessionState: SessionState
        get() {
            return sessionStateManager!!.sessionState
        }

    private var markerSyncEngine: MarkerSyncEngine? = null
    private var markReadEngine: MarkReadEngine? = null
    private var messagePreviewEngine: MessagePreviewEngine? = null
    private var messageSyncEngine: MessageSyncEngine? = null
    private var subChannelReadReceiptSyncEngine: SubChannelReadReceiptSyncEngine? = null
    private var channelReadReceiptSyncEngine: ChannelReadReceiptSyncEngine? = null
    private var objectResolverEngine: ObjectResolverEngine? = null
    private var liveReactionSyncEngine: LiveReactionSyncEngine? = null

    private var isUnreadCountEnable: Boolean = false
    private var uploadedFileAccessType: AmityFileAccessType = AmityFileAccessType.PUBLIC
    internal var logoutProcessState: LogoutProcessState = LogoutProcessState.IDLE


    /**
     * This method setup the SDK by overriding default endpoints
     *
     * @param apiKey an api key to be used with the SDK
     * @param endpoint an endpoint model to override default endpoints
     *
     */
    fun setup(
        apiKey: String,
        endpoint: AmityEndpoint,
        dbEncryption: AmityDBEncryption = AmityDBEncryption.NONE
    ): Completable {
        if (!EkoSdkInitProvider.isInitialized()) {
            return Completable.never()
        }
        EkoDatabase.init(AppContext.get(), dbEncryption)
        UserDatabase.init(AppContext.get(), dbEncryption)
        logoutProcessState = LogoutProcessState.IDLE

        val result = CompletableSubject.create()
        val database = EkoDatabase.get()
        val httpUrlDao = database.httpUrlDao()
        val mqttUrlDao = database.mqttUrlDao()
        val uploadUrlDao = database.uploadUrlDao()
        val newHttpUrl = endpoint.httpEndpoint
        val newMqttUrl = endpoint.mqttEndpoint
        val newUploadUrl = endpoint.uploadEndpoint
        val apiKeyDao = EkoDatabase.get().apiKeyDao()

        Maybe.zip(
            apiKeyDao.currentApiKey,
            httpUrlDao.currentHttpUrl,
            mqttUrlDao.currentMqttUrl,
            uploadUrlDao.currentUploadUrl,
        ) { apiKey: EkoApiKey, httpEntity: EkoHttpUrl, mqttEntity: EkoMqttUrl, uploadEntity: AmityUploadUrl ->
            val storedEndpoint =
                AmityEndpoint.CUSTOM(httpEntity.httpUrl, mqttEntity.mqttUrl, uploadEntity.uploadUrl)
            Pair(apiKey, storedEndpoint)
        }.subscribeOn(Schedulers.io())
            .doOnSuccess { setupPair: Pair<EkoApiKey, AmityEndpoint> ->
                val storedApiKey = setupPair.first.apiKey
                val storedHttpUrl = setupPair.second.httpEndpoint
                val storedMqttUrl = setupPair.second.mqttEndpoint
                val storedUploadUrl = setupPair.second.uploadEndpoint
                if (!Objects.equal(storedHttpUrl, newHttpUrl)
                    || !Objects.equal(storedMqttUrl, newMqttUrl)
                    || !Objects.equal(storedApiKey, apiKey)
                    || !Objects.equal(storedUploadUrl, newUploadUrl)
                ) {
                    AmityLog.tag(TAG).e(
                        "Setup value changed. new api key: %s new http url: %s new mqtt url: %s new upload url: %s",
                        apiKey,
                        newHttpUrl,
                        newMqttUrl,
                        newUploadUrl
                    )
                    AmityLog.tag(TAG).e("deleting user database")
                    publishLogoutEvent()
                    httpUrlDao.insert(EkoHttpUrl.create(newHttpUrl))
                    mqttUrlDao.insert(EkoMqttUrl.create(newMqttUrl))
                    uploadUrlDao.insert(AmityUploadUrl.create(newUploadUrl))
                    apiKeyDao.insert(EkoApiKey.create(apiKey))
                }
                AmityCoreSDKInitializer.initApiService()
            }
            .doOnComplete {
                logoutProcessState = LogoutProcessState.IDLE
                httpUrlDao.insert(EkoHttpUrl.create(newHttpUrl))
                mqttUrlDao.insert(EkoMqttUrl.create(newMqttUrl))
                uploadUrlDao.insert(AmityUploadUrl.create(newUploadUrl))
                apiKeyDao.insert(EkoApiKey.create(apiKey))
                AmityCoreSDKInitializer.initApiService()
            }
            .flatMapCompletable { Completable.complete() }
            .subscribe(result)

        setupSessionComponents()
        return result.hide()
    }

    fun disconnect(): Completable {
        return if (mqttClient == null) {
            Completable.complete()
        } else {
            mqttClient!!.disconnect()
        }
    }

    fun login(userId: String, sessionHandler: SessionHandler?): AmitySessionEstablisher.Builder {
        val isLegacyVersion = (sessionHandler == null)
        val sessionState = sessionStateManager?.sessionState
        if (sessionState is SessionState.Establishing) {
            throw SessionError.fromState(sessionState)
        }

        sessionHandler?.let {
            renewalManager?.sessionHandler = it
        }

        return AmitySessionEstablisher.Builder(
            userId = userId,
            appEventBus = appEventBus,
            sessionLifeCycleEventBus = sessionLifeCycleEventBus,
            isLegacyVersion = isLegacyVersion,
        )
    }

    fun loginAsVisitor(
        sessionHandler: SessionHandler
    ): AmityVisitorEstablisher.Builder {
        val sessionState = sessionStateManager?.sessionState
        if (sessionState is SessionState.Establishing) {
            throw SessionError.fromState(sessionState)
        }
        sessionHandler.let {
            renewalManager?.sessionHandler = it
        }
        return AmityVisitorEstablisher.Builder(
            appEventBus = appEventBus,
            sessionLifeCycleEventBus = sessionLifeCycleEventBus,
        )
    }

    fun publishLogoutEvent() {
        appEventBus?.publish(AppEvent.ManualLogout)
    }

    fun observeSessionState(): Flowable<SessionState> {
        return sessionStateEventBus?.observe() ?: Flowable.error(AmityException.create("This function cannot be called before setup()", null, AmityError.UNSUPPORTED))
    }

    fun observeUserUnread(): Flowable<UserUnread> {
        return ObserveUserMarker().execute().map(UserUnreadModelMapper()::map)
    }

    fun presence(): AmityPresenceService {
        return AmityPresenceService(presenceSyncEngine)
    }

    fun newUserPresenceRepository(): AmityUserPresenceRepository {
        return AmityUserPresenceRepository()
    }

    fun getAnalyticsEngine(): AnalyticsEngine? {
        return analyticsEngine
    }

    fun createTextMessage(message: EkoMessageEntity, emitter: CompletableEmitter) {
        messageSyncEngine?.addTextMessageJob(message, emitter)
    }

    fun createAttachmentMessage(message: EkoMessageEntity, emitter: CompletableEmitter, attachment: AmityMessageAttachment) {
        messageSyncEngine?.addAttachmentMessageJob(message, emitter, attachment)
    }

    fun isConsistentMode(): Boolean {
        return markerSyncEngine?.isConsistentMode() ?: true
    }

    fun isUnreadCountEnable(): Boolean {
        return isUnreadCountEnable
    }

    fun enableUnreadCount() {
        isUnreadCountEnable = true
    }

    fun markRead(subChannelId: String, segment: Int) {
        if(!isUnreadCountEnable) {
            channelReadReceiptSyncEngine?.markRead(subChannelId, segment)
        } else {
            subChannelReadReceiptSyncEngine?.markRead(subChannelId, segment)
        }
    }

    fun resolve(id: String, referenceType: ObjectResolverEngine.Companion.ReferenceType) {
        objectResolverEngine?.resolve(id, referenceType)
    }

    fun resolve(ids: List<String>, referenceType: ObjectResolverEngine.Companion.ReferenceType) {
        objectResolverEngine?.resolve(ids, referenceType)
    }

    fun setUploadedFileAccessType(type: AmityFileAccessType) {
        uploadedFileAccessType = type
    }

    fun getUploadedFileAccessType(): AmityFileAccessType {
        return uploadedFileAccessType
    }

    fun getAccessToken(): String? {
        return getCurrentAccount()?.accessToken
    }
    fun observeNetworkActivities(): Flowable<AmityNetworkActivity> {
        return networkActivityPublisher.toFlowable(BackpressureStrategy.BUFFER)
    }

    internal fun createReaction(
        domainId: LiveReactionSyncEngine.ReactionDomainId,
        referenceId: String,
        referenceType: AmityLiveReactionReferenceType,
        reactionName: String
    ) {
        liveReactionSyncEngine?.createReaction(
            domainId = domainId,
            liveReaction = AmityLiveReaction(
                referenceId = referenceId,
                referenceType = referenceType.value,
                reactionName = reactionName,
                userId = AmityCoreClient.getUserId(),
                occurredAt = DateTime.now(),
            )
        )
    }

    private fun getCurrentAccount(): EkoAccount? {
        return EkoDatabase.get().accountDao().currentAccountNow
    }

    private fun setupSessionComponents() {
        //setup session components
        //this is necessary to check if the sessionComponent instances
        //are initiated or not, so that we won't re-create mqtt
        //instances again.
        logoutProcessState = LogoutProcessState.IDLE
        if (sessionLifeCycleEventBus == null) {
            sessionLifeCycleEventBus = SessionLifeCycleEventBus()
        }
        if (appEventBus == null) {
            appEventBus = AppEventBus()
        }
        if (sessionStateEventBus == null) {
            sessionStateEventBus = SessionStateEventBus()
        }
        if (sessionStateManager == null) {
            sessionStateManager = SessionStateManager(
                appEventBus = appEventBus!!,
                sessionStateEventBus = sessionStateEventBus!!,
                sessionLifeCycleEventBus = sessionLifeCycleEventBus!!
            )
        }

        if (mqttClient == null || renewalManager == null) {
            mqttClient = AmityMqttClient(
                sessionLifeCycleEventBus = sessionLifeCycleEventBus!!,
                sessionStateEventBus = sessionStateEventBus!!
            )
            renewalManager = TokenRenewalSessionComponent(
                appEventBus = appEventBus!!,
                sessionLifeCycleEventBus = sessionLifeCycleEventBus!!,
                sessionStateEventBus = sessionStateEventBus!!
            )
            presenceSyncEngine = PresenceSyncEngine(
                sessionLifeCycleEventBus = sessionLifeCycleEventBus!!,
                sessionStateEventBus = sessionStateEventBus!!
            )
            analyticsEngine = AnalyticsEngine(
                sessionLifeCycleEventBus = sessionLifeCycleEventBus!!,
                sessionStateEventBus = sessionStateEventBus!!
            )
            AmityHttpClient(
                sessionLifeCycleEventBus = sessionLifeCycleEventBus!!,
                sessionStateEventBus = sessionStateEventBus!!
            )
            DatabaseSessionComponent(
                sessionLifeCycleEventBus = sessionLifeCycleEventBus!!,
                sessionStateEventBus = sessionStateEventBus!!
            )
            UserSettingSessionComponent(
                sessionLifeCycleEventBus = sessionLifeCycleEventBus!!,
                sessionStateEventBus = sessionStateEventBus!!
            )
            TokenWatcherSessionComponent(
                appEventBus = appEventBus!!,
                sessionLifeCycleEventBus = sessionLifeCycleEventBus!!,
                sessionStateEventBus = sessionStateEventBus!!
            )
            markerSyncEngine = MarkerSyncEngine(
                sessionLifeCycleEventBus = sessionLifeCycleEventBus!!,
                sessionStateEventBus = sessionStateEventBus!!
            )
            markReadEngine = MarkReadEngine(
                sessionLifeCycleEventBus = sessionLifeCycleEventBus!!,
                sessionStateEventBus = sessionStateEventBus!!
            )
            messagePreviewEngine = MessagePreviewEngine(
                sessionLifeCycleEventBus = sessionLifeCycleEventBus!!,
                sessionStateEventBus = sessionStateEventBus!!
            )
            messageSyncEngine = MessageSyncEngine(
                sessionLifeCycleEventBus = sessionLifeCycleEventBus!!,
                sessionStateEventBus = sessionStateEventBus!!
            )
            subChannelReadReceiptSyncEngine = SubChannelReadReceiptSyncEngine(
                sessionLifeCycleEventBus = sessionLifeCycleEventBus!!,
                sessionStateEventBus = sessionStateEventBus!!
            )
            objectResolverEngine = ObjectResolverEngine(
                sessionLifeCycleEventBus = sessionLifeCycleEventBus!!,
                sessionStateEventBus = sessionStateEventBus!!
            )
            channelReadReceiptSyncEngine = ChannelReadReceiptSyncEngine(
                sessionLifeCycleEventBus = sessionLifeCycleEventBus!!,
                sessionStateEventBus = sessionStateEventBus!!
            )
            liveReactionSyncEngine = LiveReactionSyncEngine(
                sessionLifeCycleEventBus = sessionLifeCycleEventBus!!,
                sessionStateEventBus = sessionStateEventBus!!
            )
        }
    }
}