package com.amity.socialcloud.sdk.api.social.event.query

import androidx.paging.PagingData
import com.amity.socialcloud.sdk.model.social.event.AmityEvent
import com.amity.socialcloud.sdk.model.social.event.AmityEventOriginType
import com.amity.socialcloud.sdk.model.social.event.AmityEventStatus
import com.amity.socialcloud.sdk.model.social.event.AmityEventType
import com.ekoapp.ekosdk.internal.usecase.event.GetEventPagingDataUseCase
import io.reactivex.rxjava3.core.Flowable

class AmityEventQuery internal constructor(
    private val options: AmityEventQueryOption
) {

    /**
     * Queries for events.
     * @return Flowable<PagingData<AmityEvent>>.
     */
    fun query(): Flowable<PagingData<AmityEvent>> {
        return GetEventPagingDataUseCase().execute(
            originType = options.originType,
            originId = options.originId,
            userId = options.userId,
            status = options.status,
            type = options.type,
            onlyAttendee = options.onlyAttendee,
            sortBy = options.sortBy,
            orderBy = options.orderBy
        )
    }

    class Builder {
        private var originType: AmityEventOriginType? = null
        private var originId: String? = null
        private var userId: String? = null
        private var status: AmityEventStatus? = null
        private var type: AmityEventType? = null
        private var onlyAttendee: Boolean? = null
        private var sortBy: AmityEventSortOption = AmityEventSortOption.START_TIME
        private var orderBy: AmityEventOrderOption = AmityEventOrderOption.ASCENDING

        /**
         * Specifies originType to filter events by origin type.
         * @return the same AmityEventQuery.Builder instance.
         */
        fun originType(originType: AmityEventOriginType): Builder {
            return apply { this.originType = originType }
        }

        /**
         * Specifies originId to filter events by origin ID.
         * @return the same AmityEventQuery.Builder instance.
         */
        fun originId(originId: String): Builder {
            return apply { this.originId = originId }
        }

        /**
         * Specifies userId to filter events created by specific user.
         * @return the same AmityEventQuery.Builder instance.
         */
        fun userId(userId: String): Builder {
            return apply { this.userId = userId }
        }

        /**
         * Specifies status to filter events by status.
         * @return the same AmityEventQuery.Builder instance.
         */
        fun status(status: AmityEventStatus): Builder {
            return apply { this.status = status }
        }

        /**
         * Specifies type to filter events by type (virtual or in_person).
         * @return the same AmityEventQuery.Builder instance.
         */
        fun type(type: AmityEventType): Builder {
            return apply { this.type = type }
        }

        /**
         * Filter to show only events the user has attended.
         * @return the same AmityEventQuery.Builder instance.
         */
        fun onlyAttendee(onlyAttendee: Boolean): Builder {
            return apply { this.onlyAttendee = onlyAttendee }
        }

        /**
         * Specifies the field to sort events by.
         * @param sortBy The sort option. Default is START_TIME.
         * @return the same AmityEventQuery.Builder instance.
         */
        fun sortBy(sortBy: AmityEventSortOption): Builder {
            return apply { this.sortBy = sortBy }
        }

        /**
         * Specifies the order direction for sorting.
         * @param orderBy The order option. Default is ASCENDING.
         * @return the same AmityEventQuery.Builder instance.
         */
        fun orderBy(orderBy: AmityEventOrderOption): Builder {
            return apply { this.orderBy = orderBy }
        }

        /**
         * Instantiates AmityEventQuery with built params.
         * @return new instance of AmityEventQuery.
         */
        fun build(): AmityEventQuery {
            val options = AmityEventQueryOption(
                originType = originType,
                originId = originId,
                userId = userId,
                status = status,
                type = type,
                onlyAttendee = onlyAttendee,
                sortBy = sortBy,
                orderBy = orderBy
            )
            return AmityEventQuery(options)
        }
    }
}
