package com.amity.socialcloud.sdk.chat.domain.channel

import android.util.Log
import com.amity.socialcloud.sdk.chat.data.marker.channel.ChannelMarkerRepository
import com.amity.socialcloud.sdk.chat.data.channel.reader.ChannelReaderRepository
import com.amity.socialcloud.sdk.chat.domain.message.preview.MessagePreviewGetUseCase
import com.amity.socialcloud.sdk.core.CoreClient
import com.amity.socialcloud.sdk.core.domain.ComposerUseCase
import com.amity.socialcloud.sdk.core.domain.file.ImageGetUseCase
import com.amity.socialcloud.sdk.model.chat.channel.AmityChannel

internal class ChannelComposerUseCase : ComposerUseCase() {

    fun execute(channel: AmityChannel): AmityChannel {
        return channel
                .apply(::addAvatar)
                .apply(::addUnreadCount)
                .apply(::addUnreadMention)
                .apply(::addMessagePreview)
    }

    private fun addAvatar(channel: AmityChannel): AmityChannel {
        return channel.apply {
            avatar = avatarFileId?.let { ImageGetUseCase().execute(it) } ?: avatar
        }
    }

    private fun addUnreadCount(channel: AmityChannel): AmityChannel {
        return if (!CoreClient.isUnreadCountEnable()) {
            ChannelMarkerRepository().getChannelUnread(channel.getChannelId())?.let {
                channel.apply {
                    unreadCount = it.unreadCount
                    hasMentioned = it.isMentioned
                    isMentioned = it.isMentioned
                }
            } ?: channel
        } else if (CoreClient.isConsistentMode()) {
            ChannelMarkerRepository().getChannelUnreadInfo(channel.getChannelId())?.let {
                channel.apply {
                    unreadCount = it.unreadCount
                    hasMentioned = it.isMentioned
                    subChannelsUnreadCount = it.unreadCount
                    isMentioned = it.isMentioned
                }
            } ?: channel
        } else {
            val channelMarker = ChannelMarkerRepository().getChannelMarker(channel.getChannelId())
            channel.apply {
                unreadCount = channelMarker?.unreadCount ?: 0
                hasMentioned = channelMarker?.hasMentioned ?: false
            }
        }
    }

    private fun addUnreadMention(channel: AmityChannel): AmityChannel {
        return channel.apply {
            hasDefaultSubChannelUnreadMention = ChannelReaderRepository().hasUnreadMention(channel.getChannelId())
        }
    }

    private fun addMessagePreview(channel: AmityChannel): AmityChannel {
        return channel.apply {
            messagePreview = messagePreviewId?.let { MessagePreviewGetUseCase().execute(it) }
        }
    }

}