package com.ekoapp.ekosdk.internal.data.dao;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.room.Dao;
import androidx.room.Query;
import androidx.room.Transaction;

import com.amity.socialcloud.sdk.model.core.role.AmityRoles;
import com.ekoapp.ekosdk.internal.data.UserDatabase;
import com.ekoapp.ekosdk.internal.data.model.ChannelMembershipPermissionEntity;
import com.ekoapp.ekosdk.internal.data.model.ChannelMembershipRoleEntity;
import com.ekoapp.ekosdk.internal.entity.ChannelMembershipEntity;

import org.joda.time.DateTime;

import java.util.List;

import io.reactivex.rxjava3.core.Flowable;


@Dao
public abstract class EkoChannelMembershipDao extends EkoObjectDao<ChannelMembershipEntity> implements AmityPagingDao<ChannelMembershipEntity> {

    private final EkoChannelRoleDao channelRoleDao;
    private final EkoChannelPermissionDao channelPermissionDao;

    EkoChannelMembershipDao() {
        UserDatabase db = UserDatabase.get();
        channelRoleDao = db.channelRoleDao();
        channelPermissionDao = db.channelPermissionDao();
    }

    @Query("SELECT *" +
            " from channel_membership" +
            " where id = :id" +
            " LIMIT 1")
    @Nullable
    public abstract ChannelMembershipEntity getByIdNow(@NonNull String id);

    @Query("SELECT *" +
            " from channel_membership" +
            " where id IN (:ids)" +
            " LIMIT 1")
    @NonNull
    public abstract List<ChannelMembershipEntity> getByIdsNow(@NonNull List<String> ids);

    @Query("UPDATE channel_membership set membership = :membership, isBanned = 1 where channelId = :channelId and userId = :userId")
    public abstract void updateMembership(String channelId, String userId, String membership);

    @Query("UPDATE channel_membership set readToSegment = :readToSegment where channelId = :channelId and userId = :userId and readToSegment < :readToSegment")
    public abstract void updateReadToSegment(String channelId, String userId, int readToSegment);

    @Query("UPDATE channel_membership set lastMentionedSegment = :lastMentionedSegment where channelId = :channelId and userId = :userId and lastMentionedSegment < :lastMentionedSegment")
    public abstract void updateLastMentionedSegment(String channelId, String userId, int lastMentionedSegment);

    @Query("UPDATE channel_membership set isMuted = :isMuted where channelId = :channelId and userId = :userId")
    public abstract void updateMemberIsMuted(String channelId, String userId, boolean isMuted);

    @Query("DELETE from channel_membership")
    public abstract void deleteAll();

    @Query("DELETE from channel_membership where channelId = :channelId")
    public abstract void deleteAllFromChannel(String channelId);

    @Query("SELECT *" +
            " from channel_membership" +
            " where channelId = :channelId" +
            " and userId = :userId" +
            " LIMIT 1")
    abstract Flowable<ChannelMembershipEntity> getByChannelIdAndUserIdImpl(String channelId, String userId);

    public Flowable<ChannelMembershipEntity> getByChannelIdAndUserId(String channelId, String userId) {
        return getByChannelIdAndUserIdImpl(channelId, userId);
    }

    @Query("SELECT *" +
            " from channel_membership" +
            " where channelId = :channelId")
    abstract Flowable<List<ChannelMembershipEntity>> getByChannelIdImpl(String channelId);

    public Flowable<List<ChannelMembershipEntity>> getByChannelId(String channelId) {
        return getByChannelIdImpl(channelId);
    }

    @Query("SELECT *" +
            " from channel_membership" +
            " where channelId = :channelId" +
            " and userId = :userId" +
            " LIMIT 1")
    public abstract ChannelMembershipEntity getByChannelIdAndUserIdNow(String channelId, String userId);

    @Query("SELECT *" +
            " from channel_membership" +
            " where channelId = :channelId")
    public abstract List<ChannelMembershipEntity> getByChannelIdNow(String channelId);

    @Query("SELECT *" +
            " from channel_membership" +
            " where channel_membership.channelId = :channelId" +
            " and channel_membership.membership in (:memberships)" +
            " and case when :isFilterMuted then channel_membership.isMuted = :isMuted" +
            " else isMuted is not null end" + // always true
            " and case when :isFilterByRoles then channel_membership.userId in (SELECT userId from channel_role where channelId = :channelId and roleName in (:roles))" +
            " else userId is not null end" + // always true
            " and channel_membership.updatedAt > :now" +
            " and channel_membership.userId not in " +
            "(" +
            "SELECT amity_paging_id.id" +
            " from amity_paging_id" +
            " where amity_paging_id.hash = (:hash)" +
            " and amity_paging_id.nonce = (:nonce) " +
            ")" +
            " order by channel_membership.updatedAt  desc" +
            " limit 1"
    )
    abstract Flowable<ChannelMembershipEntity> getLatestChannelMembershipImpl(String channelId,
                                                                              String[] memberships,
                                                                              Boolean isFilterMuted,
                                                                              Boolean isMuted,
                                                                              Boolean isFilterByRoles,
                                                                              String[] roles,
                                                                              int hash,
                                                                              int nonce,
                                                                              DateTime now);

    public Flowable<ChannelMembershipEntity> getLatestChannelMembership(String channelId,
                                                                        List<String> memberships,
                                                                        Boolean isMuted,
                                                                        AmityRoles roles,
                                                                        int hash,
                                                                        int nonce,
                                                                        DateTime now) {
        return getLatestChannelMembershipImpl(channelId,
                memberships.toArray(new String[0]),
                isMuted != null,
                isMuted,
                !roles.get().isEmpty(),
                roles.get().toArray(new String[0]),
                hash,
                nonce,
                now);
    }

    @Transaction
    @Override
    public void insert(ChannelMembershipEntity membership) {
        super.insert(membership);
        EkoRoleDao.update(membership, channelRoleDao, ChannelMembershipRoleEntity::create);
        EkoPermissionDao.update(membership, channelPermissionDao, ChannelMembershipPermissionEntity::create);
    }

    @Transaction
    @Override
    public void insert(List<? extends ChannelMembershipEntity> memberships) {
        super.insert(memberships);
        EkoRoleDao.update(memberships, channelRoleDao, ChannelMembershipRoleEntity::create);
        EkoPermissionDao.update(memberships, channelPermissionDao, ChannelMembershipPermissionEntity::create);
    }

    @Override
    public void update(ChannelMembershipEntity membership) {
        super.update(membership);
        EkoRoleDao.update(membership, channelRoleDao, ChannelMembershipRoleEntity::create);
        EkoPermissionDao.update(membership, channelPermissionDao, ChannelMembershipPermissionEntity::create);
    }

    @Transaction //dummy update, for triggering user update tied to channel member.
    public void updateUser(String userId) {
        updateUserImpl(userId);
    }

    @Query("UPDATE channel_membership set userId = :userId where userId = :userId")
    abstract void updateUserImpl(String userId);
}
