package com.amity.socialcloud.sdk.core.data.room

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import co.amity.rxbridge.toRx3
import com.amity.socialcloud.sdk.model.video.room.AmityRoom
import com.amity.socialcloud.sdk.model.video.room.AmityRoomSortOption
import com.amity.socialcloud.sdk.model.video.room.AmityRoomStatus
import com.amity.socialcloud.sdk.model.video.room.AmityRoomType
import com.amity.socialcloud.sdk.dto.video.EkoRoomBroadcastDataDto
import com.amity.socialcloud.sdk.video.data.room.RoomModelMapper
import com.amity.socialcloud.sdk.video.data.room.RoomModerationModelMapper
import com.ekoapp.ekosdk.EkoObjectRepository
import com.ekoapp.ekosdk.internal.paging.DynamicQueryStreamPagerCreator
import com.ekoapp.ekosdk.internal.token.DynamicQueryStreamQueryToken
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import com.ekoapp.ekosdk.internal.api.socket.request.CreateRoomRequest
import com.ekoapp.ekosdk.internal.api.socket.request.UpdateRoomRequest
import com.amity.socialcloud.sdk.video.domain.room.compose.RoomComposerUseCase
import com.google.gson.JsonObject

internal class RoomRepository : EkoObjectRepository() {

    private fun defaultPageSize() = DEFAULT_PAGE_SIZE

    @OptIn(ExperimentalPagingApi::class)
    fun getRoomCollectionPagingData(
        statuses: Array<AmityRoomStatus>?,
        types: Array<AmityRoomType>?,
        isDeleted: Boolean?,
        sortBy: AmityRoomSortOption?
    ): Flowable<PagingData<AmityRoom>> {
        val statusList = formatStatuses(statuses)
        val typeList = formatTypes(types)
        val sortKey = sortBy?.apiKey
        val mediator = RoomMediator(
            statuses = statusList,
            types = typeList,
            isDeleted = isDeleted,
            sortBy = sortKey
        )
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = defaultPageSize(),
                enablePlaceholders = false
            ),
            dynamicQueryStreamMediator = mediator,
            pagingSourceFactory = {
                RoomLocalDataStore().getRoomsPagingSource(
                    statuses = statusList,
                    types = typeList,
                    isDeleted = isDeleted,
                    sortBy = sortKey
                )
            },
            modelMapper = RoomModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    fun observeRoom(roomId: String): Flowable<AmityRoom> =
        RoomLocalDataStore().observeRoom(roomId)
            .map(RoomModelMapper()::map)
            .map { RoomComposerUseCase().execute(it) }

    fun getRoomsByIds(roomIds: List<String>): List<AmityRoom> =
        RoomLocalDataStore().getRoomsByIds(roomIds)
            .map(RoomModelMapper()::map)
            .map { RoomComposerUseCase().execute(it) }

    fun fetchRoom(roomId: String): Completable {
        return RoomRemoteDataStore().getRoom(roomId)
            .flatMapCompletable { RoomQueryPersister().persist(it) }
    }

    fun createRoom(
        title: String,
        description: String?,
        thumbnailFileId: String?,
        metadata: JsonObject?,
        channelEnabled: Boolean?,
        parentRoomId: String?,
        participants: List<String>
    ): Single<String> {
        val request = CreateRoomRequest(
            title = title,
            description = description,
            thumbnailFileId = thumbnailFileId,
            metadata = metadata,
            channelEnabled = channelEnabled,
            parentRoomId = parentRoomId,
            participants = participants
        )
        return RoomRemoteDataStore().createRoom(request)
            .flatMap { dto ->
                RoomListQueryPersister().persist(dto)
                    .andThen(Single.just(dto.rooms?.firstOrNull()?.roomId ?: ""))
            }
    }

    fun updateRoom(
        roomId: String,
        title: String?,
        description: String?,
        thumbnailFileId: String?,
        metadata: JsonObject?,
        channelEnabled: Boolean?
    ): Single<String> {
        val request = UpdateRoomRequest(
            title = title,
            description = description,
            thumbnailFileId = thumbnailFileId,
            metadata = metadata,
            channelEnabled = channelEnabled
        )
        return RoomRemoteDataStore().updateRoom(roomId, request)
            .flatMap { dto ->
                RoomQueryPersister().persist(dto)
                    .andThen(Single.just(dto.rooms?.firstOrNull()?.roomId ?: roomId))
            }
    }

    fun deleteRoom(roomId: String): Completable =
        RoomRemoteDataStore().deleteRoom(roomId).ignoreElement()

    fun stopRoom(roomId: String): Single<String> =
        RoomRemoteDataStore().stopRoom(roomId)
            .flatMap { dto ->
                RoomQueryPersister().persist(dto)
                    .andThen(Single.just(dto.rooms?.firstOrNull()?.roomId ?: roomId))
            }

    fun getRecordedUrls(roomId: String): Single<List<String>> =
        RoomRemoteDataStore().getRoomRecordedUrls(roomId).map { it.recordedUrls ?: emptyList() }

    fun getBroadcasterData(roomId: String): Single<EkoRoomBroadcastDataDto> =
        RoomRemoteDataStore().getBroadcasterData(roomId)

    fun getLatestRoom(statuses: List<String>): Flowable<AmityRoom> =
        RoomLocalDataStore().getLatestRoom(statuses)
            .map(RoomModelMapper()::map)
            .map { RoomComposerUseCase().execute(it) }

    fun getRoomModerationByRoom(roomId: String) =
        RoomModerationLocalDataStore().getModerationByRoomId(roomId)?.let { RoomModerationModelMapper().map(it) }

    private fun formatStatuses(statuses: Array<AmityRoomStatus>?): List<String> =
        (statuses?.takeIf { it.isNotEmpty() } ?: arrayOf(AmityRoomStatus.IDLE)).map { it.apiKey }

    private fun formatTypes(types: Array<AmityRoomType>?): List<String>? =
        types?.takeIf { it.isNotEmpty() }?.map { it.apiKey }

    // For future token usage if needed externally
    internal fun buildQueryToken(statuses: List<String>): DynamicQueryStreamQueryToken =
        DynamicQueryStreamQueryToken(dynamicQueryStreamKeyCreator = com.ekoapp.ekosdk.internal.keycreator.RoomKeyCreator(statuses, null, null, null), next = null, previous = null, primaryKeys = emptyList())
}
