package com.amity.socialcloud.sdk.core.session

import com.amity.socialcloud.sdk.core.domain.session.GetCurrentAccountUseCase
import com.amity.socialcloud.sdk.core.domain.session.RenewTokenUseCase
import com.amity.socialcloud.sdk.core.session.eventbus.AppEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionLifeCycleEventBus
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.schedulers.Schedulers
import org.joda.time.DateTime

class AccessTokenRenewalImpl(
    private val appEventBus: AppEventBus,
    private val sessionLifeCycleEventBus: SessionLifeCycleEventBus,
    private val onRenewTokenSuccess: () -> Unit,
    private val onRenewTokenFailed: (throwable: Throwable) -> Unit,
    private val onUnableToRetrieveAuthToken: () -> Unit,

    ) : AccessTokenRenewal {

    var invalidated = false

    override fun renew() {
        if (this.invalidated) {
            return
        }
        invalidate()
        refreshToken()
            .doOnComplete {
                onRenewTokenSuccess()
            }
            .doOnError {
                onRenewTokenFailed(it)
            }
            .subscribeOn(Schedulers.io())
            .subscribe()
    }

    override fun renewWithAuthToken(authToken: String) {
        if (this.invalidated) {
            return
        }
        invalidate()
        refreshToken(authToken)
            .doOnComplete {
                onRenewTokenSuccess()
            }
            .doOnError {
                onRenewTokenFailed(it)
            }
            .subscribeOn(Schedulers.io())
            .subscribe()
    }

    override fun renewWithAuthSignature(authSignature: String, authSignatureExpiresAt: DateTime) {
        if (this.invalidated) {
            return
        }
        invalidate()
        refreshToken()
                .doOnComplete {
                    onRenewTokenSuccess()
                }
                .doOnError {
                    onRenewTokenFailed(it)
                }
                .subscribeOn(Schedulers.io())
                .subscribe()
    }

    override fun unableToRetrieveAuthToken() {
        onUnableToRetrieveAuthToken()
    }

    fun invalidate() {
        this.invalidated = true
    }

    private fun refreshToken(
        authToken: String? = null,
        authSignature: String? = null,
        authSignatureExpiresAt: DateTime? = null,
    ): Completable {
        return GetCurrentAccountUseCase().execute()
            .flatMap {
                RenewTokenUseCase().execute(
                    appEventBus = appEventBus,
                    sessionLifeCycleEventBus = sessionLifeCycleEventBus,
                    ekoAccount = it,
                    authToken = authToken,
                    authSignature = authSignature,
                    authSignatureExpiresAt = authSignatureExpiresAt,
                )
            }
            .subscribeOn(Schedulers.io())
            .ignoreElement()
    }
}