package com.amity.socialcloud.sdk.core

import com.amity.socialcloud.sdk.chat.data.channel.ChannelRepository
import com.amity.socialcloud.sdk.chat.data.message.preview.MessagePreviewRepository
import com.amity.socialcloud.sdk.chat.data.subchannel.SubChannelRepository
import com.amity.socialcloud.sdk.chat.domain.message.preview.MessageEventToMessagePreviewUseCase
import com.amity.socialcloud.sdk.chat.domain.message.preview.MessagePreviewSaveUseCase
import com.amity.socialcloud.sdk.chat.domain.message.preview.MessagePreviewUpdatedUseCase
import com.amity.socialcloud.sdk.core.session.component.SessionComponent
import com.amity.socialcloud.sdk.core.session.eventbus.MessagePreviewEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionLifeCycleEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionStateEventBus
import com.amity.socialcloud.sdk.core.session.model.SessionState
import com.amity.socialcloud.sdk.model.chat.settings.AmityMessagePreviewSetting
import com.ekoapp.ekosdk.internal.data.model.EkoAccount
import com.ekoapp.ekosdk.internal.usecase.user.GetChatSettingsUseCase
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.schedulers.Schedulers

internal class MessagePreviewEngine(
	sessionLifeCycleEventBus: SessionLifeCycleEventBus,
	sessionStateEventBus: SessionStateEventBus
) : SessionComponent(sessionLifeCycleEventBus, sessionStateEventBus) {
	
	private var isActive = sessionStateEventBus.getCurrentEvent() == SessionState.Established
	private var messagePreviewSetting: AmityMessagePreviewSetting? = null
	
	init {
		if (isActive) {
			fetchChatSettings()
		}
		MessagePreviewEventBus
			.observe()
			.flatMapCompletable { event  ->
				handleMessagePreviewEvent(event).onErrorComplete()
			}
				.subscribeOn(Schedulers.io())
				.subscribe()
	}
	
	private fun handleMessagePreviewEvent(event: MessagePreviewEvent): Completable {
		return if (event is MessagePreviewEvent.MessagePreviewSettingChange) {
			when (event.messagePreviewSetting) {
				AmityMessagePreviewSetting.NO_MESSAGE_PREVIEW -> {
					Completable.fromCallable { MessagePreviewRepository().clearAllMessagePreviews() }
				}
				AmityMessagePreviewSetting.MESSAGE_PREVIEW_NOT_INCLUDE_DELETED -> {
					Completable.fromCallable { MessagePreviewRepository().clearDeletedMessagePreviews() }
				}
				else -> {
					Completable.complete()
				}
			}
		} else if (isActive && messagePreviewSetting != AmityMessagePreviewSetting.NO_MESSAGE_PREVIEW) {
			when (event) {
				is MessagePreviewEvent.MessageCreated -> handleMessageCreatedEvent(event)
				is MessagePreviewEvent.MessageUpdated -> handleMessageUpdated(event)
				is MessagePreviewEvent.MessageDeleted -> handleMessageDeleted(event)
				is MessagePreviewEvent.SubChannelUpdated -> handleSubChannelUpdated(event)
				is MessagePreviewEvent.SubChannelDeleted -> handleSubChannelDeleted(event)
				else -> Completable.complete()
			}
		} else {
			Completable.complete()
		}
	}
	
	private fun handleMessageCreatedEvent(event: MessagePreviewEvent.MessageCreated ): Completable {
		return MessageEventToMessagePreviewUseCase().execute(event)?.let(MessagePreviewSaveUseCase()::execute) ?: Completable.complete()
	}
	
	private fun handleMessageUpdated(event: MessagePreviewEvent.MessageUpdated): Completable {
		return MessageEventToMessagePreviewUseCase().execute(event)?.let(MessagePreviewUpdatedUseCase()::execute) ?: Completable.complete()
	}
	
	private fun handleMessageDeleted(event: MessagePreviewEvent.MessageDeleted): Completable {
		return messagePreviewSetting?.let { setting ->
			when(setting) {
				AmityMessagePreviewSetting.MESSAGE_PREVIEW_INCLUDE_DELETED -> {
					handleMessageUpdated(MessagePreviewEvent.MessageUpdated(event.message, event.subChannel))
				}
				AmityMessagePreviewSetting.MESSAGE_PREVIEW_NOT_INCLUDE_DELETED -> {
					fetchSubChannelIfNeeded(event)
				}
				AmityMessagePreviewSetting.NO_MESSAGE_PREVIEW -> {
					Completable.complete()
				}
			}
		} ?: Completable.complete()
	}
	
	private fun handleSubChannelUpdated(event: MessagePreviewEvent.SubChannelUpdated ): Completable {
		val messagePreviewRepository = MessagePreviewRepository()
		return event.subChannel.subChannelId?.let(messagePreviewRepository::getMessagePreviewBySubChannelId)?.let { cache ->
			if (cache.subChannelUpdatedAt?.isBefore(event.subChannel.updatedAt) == true && event.subChannel.name != null) {
				messagePreviewRepository.updateSubChannelInfo(event.subChannel.subChannelId, event.subChannel.name, event.subChannel.updatedAt)
					.andThen(
						Completable.fromAction {
							SubChannelRepository()
								.notifyChanges(event.subChannel.subChannelId)
						}.onErrorComplete()
					)
					.andThen (
						Completable.fromAction {
							event.subChannel.channelPublicId?.let(messagePreviewRepository::getMessagePreviewByChannelId)?.let { channelCache ->
								if (channelCache.subChannelId == event.subChannel.subChannelId) {
									ChannelRepository().notifyChanges(event.subChannel.channelPublicId)
								} else {
									Completable.complete()
								}
							}
						}.onErrorComplete()
					)
			} else {
				Completable.complete()
			}
		} ?: Completable.complete()
	}
	
	private fun handleSubChannelDeleted(event: MessagePreviewEvent.SubChannelDeleted ): Completable {
		return messagePreviewSetting?.let { setting ->
			when(setting) {
				AmityMessagePreviewSetting.MESSAGE_PREVIEW_NOT_INCLUDE_DELETED, AmityMessagePreviewSetting.MESSAGE_PREVIEW_INCLUDE_DELETED -> {
					event.subChannel.channelPublicId?.let(MessagePreviewRepository()::getMessagePreviewByChannelId)?.let { channelCache ->
						if (channelCache.subChannelId == event.subChannel.subChannelId) {
							ChannelRepository().fetchAndSave(event.subChannel.channelPublicId)
						} else {
							Completable.complete()
						}
					}
				}
				else -> Completable.complete()
			}
		} ?: Completable.complete()
	}
	
	private fun fetchSubChannelIfNeeded(event: MessagePreviewEvent.MessageDeleted): Completable {
		val messagePreviewRepository = MessagePreviewRepository()
		val cache = messagePreviewRepository.getMessagePreview(event.message.messageId)
		return if (cache != null) {
			val channelCache = messagePreviewRepository.getMessagePreviewByChannelId(event.message.channelId)
			messagePreviewRepository.deleteMessagePreview(event.message.messageId)
			SubChannelRepository()
				.fetchAndSave(event.message.subChannelId)
				.onErrorComplete()
				.andThen(
					if (channelCache?.messagePreviewId == event.message.messageId) {
						ChannelRepository()
							.fetchAndSave(event.message.channelId)
							.onErrorComplete()
					} else {
						Completable.complete()
					}
				)
		} else {
			Completable.complete()
		}
	}
	
	private fun fetchChatSettings() {
		GetChatSettingsUseCase().execute()
				.distinctUntilChanged()
				.doOnNext { messagePreviewSetting = it.messagePreviewSetting() }
				.subscribeOn(Schedulers.io())
				.subscribe()
	}
	
	override fun onSessionStateChange(sessionState: SessionState) {
		when(sessionState) {
			SessionState.Established -> {
				isActive = true
				GetChatSettingsUseCase().execute()
					.distinctUntilChanged()
					.doOnNext { messagePreviewSetting = it.messagePreviewSetting() }
					.subscribeOn(Schedulers.io())
					.subscribe()
			}
			else -> {
				isActive = false
			}
		}
	}
	
	override fun establish(account: EkoAccount) {
		isActive = true
		fetchChatSettings()
	}
	
	override fun destroy() {
		isActive = false
	}
	
	override fun handleTokenExpire() {
		isActive = false
	}
}