package com.amity.socialcloud.sdk.social.data.storytarget

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import co.amity.rxbridge.toRx3
import com.amity.socialcloud.sdk.api.social.storytarget.AmityGlobalStoryTargetsQueryOption
import com.amity.socialcloud.sdk.common.AmityObjectRepository
import com.amity.socialcloud.sdk.common.ModelMapper
import com.amity.socialcloud.sdk.entity.social.story.StoryTargetEntity
import com.amity.socialcloud.sdk.model.social.story.AmityStory
import com.amity.socialcloud.sdk.model.social.story.AmityStoryTarget
import com.amity.socialcloud.sdk.social.data.story.StoryQueryPersister
import com.amity.socialcloud.sdk.social.data.storytarget.paging.StoryTargetMediator
import com.amity.socialcloud.sdk.social.data.storytarget.singlepage.StoryTargetsSinglePageMediator
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import com.ekoapp.ekosdk.internal.paging.DynamicQueryStreamPagerCreator
import com.ekoapp.ekosdk.internal.paging.SinglePagePagerCreator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import org.joda.time.DateTime


@OptIn(ExperimentalPagingApi::class)
internal class StoryTargetRepository : AmityObjectRepository<StoryTargetEntity, AmityStoryTarget>() {

    internal fun getDefaultPageSize(): Int {
        return DEFAULT_PAGE_SIZE
    }

    override fun fetchAndSave(objectId: String): Completable {
        return StoryTargetRemoteDataStore().getStoryTarget(
            targetType = objectId.substringBefore("/"),
            targetId = objectId.substringAfter("/")
        )
            .flatMapCompletable {
                StoryQueryPersister().persist(it)
            }
    }

    override fun queryFromCache(objectId: String): StoryTargetEntity? {
        return StoryTargetLocalDataStore().getStoryTargetById(objectId)
    }

    override fun mapper(): ModelMapper<StoryTargetEntity, AmityStoryTarget> {
        return StoryTargetModelMapper()
    }

    override fun observeFromCache(objectId: String): Flowable<StoryTargetEntity> {
        return StoryTargetLocalDataStore().observeStoryTarget(objectId)
    }

    fun getStoryTarget(targetType: AmityStory.TargetType, targetId: String): AmityStoryTarget? {
        return StoryTargetLocalDataStore().getStoryTarget(targetType, targetId)?.let {
            mapper().map(it)
        }
    }

    fun getStoryTargets(targets: List<Pair<AmityStory.TargetType, String>>): Flowable<List<AmityStoryTarget>> {
        val pagerCreator = SinglePagePagerCreator(
            mediator = StoryTargetsSinglePageMediator(targets = targets),
            domainDatasource = StoryTargetLocalDataStore().getStoryTargets(targets),
            modelMapper = StoryTargetModelMapper()
        )

        return pagerCreator.create()
    }

    fun updateStoryTargetLocalLastStoryExpiresAt(
        targetType: AmityStory.TargetType,
        targetId: String,
        localLastStoryExpiresAt: DateTime?
    ): Completable {
        return if (StoryTargetLocalDataStore().getStoryTarget(targetType, targetId) != null) {
            StoryTargetLocalDataStore().updateStoryTargetLocalLastStoryExpiresAt(
                targetType = targetType,
                targetId = targetId,
                localLastStoryExpiresAt = localLastStoryExpiresAt
            )
        } else {
            StoryTargetLocalDataStore().createStoryTarget(
                targetType = targetType,
                targetId = targetId,
                lastStoryExpiresAt = null,
                localLastStoryExpiresAt = localLastStoryExpiresAt,
                lastStorySeenExpiresAt = null,
                localLastStorySeenExpiresAt = null
            )
        }
    }

    fun updateStoryTargetLastStoryExpiresAt(
        targetType: AmityStory.TargetType,
        targetId: String,
        lastStoryExpiresAt: DateTime?
    ): Completable {
        return StoryTargetLocalDataStore().updateStoryTargetLastStoryExpiresAt(
            targetType,
            targetId,
            lastStoryExpiresAt
        )
    }

    fun updateStoryTargetLocalLastStorySeenExpiresAt(
        targetType: AmityStory.TargetType,
        targetId: String,
        localLastStorySeenExpiresAt: DateTime?
    ): Completable {
        return StoryTargetLocalDataStore().updateStoryTargetLocalLastStorySeenExpiresAt(
            targetType = targetType,
            targetId = targetId,
            localLastStorySeenExpiresAt = localLastStorySeenExpiresAt
        )
    }
    fun updateStoryTargetHasUnseen(
        targetType: AmityStory.TargetType,
        targetId: String,
        hasUnseen: Boolean,
        localSortingDate: DateTime?
    ): Completable {
        return StoryTargetLocalDataStore().updateStoryTargetLocalSortingDate(
            targetType = targetType,
            targetId = targetId,
            hasUnseen = hasUnseen,
            localSortingDate = localSortingDate
        )
    }

    fun getGlobalFeed(
        option: AmityGlobalStoryTargetsQueryOption
    ): Flowable<PagingData<AmityStoryTarget>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = true
            ),
            dynamicQueryStreamMediator = StoryTargetMediator(
                option = option
            ),
            pagingSourceFactory = {
                StoryTargetLocalDataStore().getStoryTargetPagingSource(
                    option = option
                )
            },
            modelMapper = StoryTargetModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    fun getLatestStoryTargets(
        option: AmityGlobalStoryTargetsQueryOption,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int
    ): Flowable<AmityStoryTarget> {
        return StoryTargetLocalDataStore().getLatestStoryTargets(
            option,
            dynamicQueryStreamKeyCreator,
            nonce
        ).map {
            StoryTargetModelMapper().map(it)
        }
    }

}