package com.amity.socialcloud.sdk.chat.data.message

import com.amity.socialcloud.sdk.chat.data.channel.ChannelLocalDataStore
import com.amity.socialcloud.sdk.chat.data.channel.reader.ChannelReaderRepository
import com.amity.socialcloud.sdk.chat.data.message.flag.MessageFlagEntityMapper
import com.amity.socialcloud.sdk.chat.data.message.flag.MessageFlagLocalDataStore
import com.amity.socialcloud.sdk.chat.data.subchannel.SubChannelLocalDataStore
import com.amity.socialcloud.sdk.common.QueryPersister
import com.amity.socialcloud.sdk.core.data.file.FileEntityMapper
import com.amity.socialcloud.sdk.core.data.file.FileLocalDataStore
import com.amity.socialcloud.sdk.core.data.reaction.ReactionLocalDataStore
import com.amity.socialcloud.sdk.core.data.user.UserEntityMapper
import com.amity.socialcloud.sdk.core.data.user.UserLocalDataStore
import com.amity.socialcloud.sdk.core.data.user.flag.UserFlagEntityMapper
import com.amity.socialcloud.sdk.core.data.user.flag.UserFlagLocalDataStore
import com.amity.socialcloud.sdk.model.core.reaction.AmityReactionReferenceType
import com.ekoapp.ekosdk.internal.api.dto.EkoMentioneesDto
import com.ekoapp.ekosdk.internal.api.dto.EkoMessageDto
import com.ekoapp.ekosdk.internal.api.dto.EkoUserDto
import com.ekoapp.ekosdk.internal.api.dto.MessageQueryDto
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single

internal class MessageEventPersister : QueryPersister<MessageQueryDto>() {

    override fun persist(dto: MessageQueryDto): Completable {
        return FileLocalDataStore().saveFiles(FileEntityMapper().map(dto.files))
            .andThen(UserLocalDataStore().saveUsers(UserEntityMapper().map(dto.users)))
            .andThen(UserFlagLocalDataStore().saveFlags(UserFlagEntityMapper().map(dto.users)))
            .andThen(MessageLocalDataStore().saveMessages(MessageEntityMapper().map(dto.messages)))
            .andThen(MessageFlagLocalDataStore().saveFlags(MessageFlagEntityMapper().map(dto.messages)))
            .andThen(updateLastMentionedSegment(dto.messages))
            .andThen(persistReactor(dto.messages, dto.users))
            .andThen(updateChannelLastActivity(dto.messages))
            .andThen(updateSubChannelLastActivity(dto.messages))
    }

    private fun updateLastMentionedSegment(messages: List<EkoMessageDto>): Completable {
        return Flowable.fromIterable(messages)
            .flatMapCompletable {
                ChannelReaderRepository().updateLastMentionedSegment(
                    channelId = it.channelId,
                    subChannelId = it.subChannelId,
                    mentionees = it.mentionees?.map { mention -> EkoMentioneesDto(type = mention.type, userIds = mention.userPublicIds) } ?: emptyList(),
                    segment = it.segment
                )
            }
    }

    private fun persistReactor(
        messages: List<EkoMessageDto>,
        users: List<EkoUserDto>
    ): Completable {
        return Flowable.fromIterable(messages)
            .flatMapCompletable { message ->
                Single.fromCallable {
                    val reactor = message.latestReaction
                    reactor != null
                }.flatMapCompletable { hasReactor ->
                    if (!hasReactor) {
                        Completable.complete()
                    } else {
                        val reactor = message.latestReaction
                        val user = users.find { it.userId == reactor.userId }
                        when (reactor.eventName) {
                            "add" -> ReactionLocalDataStore().addReaction(
                                reactionId = reactor.reactionId,
                                referenceType = AmityReactionReferenceType.MESSAGE,
                                referenceId = message.messageId,
                                reactionName = reactor.reactionName,
                                userId = reactor.userId,
                                userDisplayName = user?.displayName,
                                createdAt = reactor.createdAt,
                                updateAt = reactor.updatedAt
                            )
                            "remove" -> ReactionLocalDataStore().removeReaction(
                                AmityReactionReferenceType.MESSAGE,
                                message.messageId,
                                reactor.reactionName,
                                reactor.userId
                            )
                            else -> Completable.complete()
                        }
                    }
                }
            }
    }

    private fun updateChannelLastActivity(messages: List<EkoMessageDto>): Completable {
        return Single.fromCallable {
            messages.map {
                it.channelId
            }.toSet()
        }.flatMapCompletable { channels ->
            Flowable.fromIterable(channels)
                .flatMapCompletable { channelId ->
                    ChannelLocalDataStore().updateLastActivity(channelId)
                }
        }
    }

    private fun updateSubChannelLastActivity(messages: List<EkoMessageDto>): Completable {
        return Single.fromCallable {
            messages.map {
                it.subChannelId
            }.toSet()
        }.flatMapCompletable { subChannels ->
            Flowable.fromIterable(subChannels)
                .flatMapCompletable { subChannelId ->
                    SubChannelLocalDataStore().updateLastActivity(subChannelId)
                }
        }
    }

}