package com.amity.socialcloud.sdk.chat.data.channel

import androidx.paging.ExperimentalPagingApi
import androidx.paging.PagingConfig
import androidx.paging.PagingData
import co.amity.rxbridge.toRx3
import com.amity.socialcloud.sdk.api.chat.channel.create.AmityChannelCreator
import com.amity.socialcloud.sdk.api.core.AmityCoreClient
import com.amity.socialcloud.sdk.chat.data.channel.membership.ChannelMembershipRepository
import com.amity.socialcloud.sdk.chat.data.channel.paging.ChannelMediator
import com.amity.socialcloud.sdk.chat.data.channel.singlepage.ChannelListMediator
import com.amity.socialcloud.sdk.chat.data.marker.channel.ChannelMarkerRepository
import com.amity.socialcloud.sdk.common.AmityObjectRepository
import com.amity.socialcloud.sdk.common.ModelMapper
import com.amity.socialcloud.sdk.core.CoreClient
import com.amity.socialcloud.sdk.core.MarkerSyncEngine
import com.amity.socialcloud.sdk.model.chat.channel.AmityChannel
import com.amity.socialcloud.sdk.model.chat.channel.AmityChannelFilter
import com.amity.socialcloud.sdk.model.chat.channel.ChannelCreateOption
import com.amity.socialcloud.sdk.model.chat.channel.ChannelUpdateOption
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.tag.AmityTags
import com.ekoapp.ekosdk.internal.EkoChannelEntity
import com.ekoapp.ekosdk.internal.api.dto.ChannelQueryDto
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.keycreator.DynamicQueryStreamKeyCreator
import com.ekoapp.ekosdk.internal.paging.DynamicQueryStreamPagerCreator
import com.ekoapp.ekosdk.internal.paging.SinglePagePagerCreator
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single

@OptIn(ExperimentalPagingApi::class)
internal class ChannelRepository : AmityObjectRepository<EkoChannelEntity, AmityChannel>() {

    override fun fetchAndSave(objectId: String): Completable {
        return ChannelRemoteDataStore().getChannel(objectId)
            .flatMapCompletable { dto ->
                fetchChannelMarker(dto)
                    .andThen(persistChannels(dto))
            }
    }

    override fun queryFromCache(objectId: String): EkoChannelEntity? {
        return ChannelLocalDataStore().getChannel(objectId)
    }

    override fun mapper(): ModelMapper<EkoChannelEntity, AmityChannel> {
        return ChannelModelMapper()
    }

    override fun observeFromCache(objectId: String): Flowable<EkoChannelEntity> {
        return ChannelLocalDataStore().observeChannel(objectId)
    }

    private fun getDefaultPageSize(): Int {
        return DEFAULT_PAGE_SIZE
    }

    fun getChannelPagingData(
        types: Set<AmityChannel.Type>,
        filter: AmityChannelFilter,
        includingTags: AmityTags,
        excludingTags: AmityTags,
        isDeleted: Boolean?,
    ): Flowable<PagingData<AmityChannel>> {
        val pagerCreator = DynamicQueryStreamPagerCreator(
            pagingConfig = PagingConfig(
                pageSize = getDefaultPageSize(),
                enablePlaceholders = true
            ),
            dynamicQueryStreamMediator = ChannelMediator(
                types = types,
                filter = filter,
                includingTags = includingTags,
                excludingTags = excludingTags,
                isDeleted = isDeleted,
            ),
            pagingSourceFactory = {
                ChannelLocalDataStore().getChannelPagingSource(
                    types = types,
                    filter = filter,
                    includingTags = includingTags,
                    excludingTags = excludingTags,
                    isDeleted = isDeleted,
                )
            },
            modelMapper = ChannelModelMapper()
        )
        return pagerCreator.create().toRx3()
    }

    fun createChannel(
        creationType: AmityChannelCreator.CreationType,
        option: ChannelCreateOption
    ): Single<AmityChannel> {
        return ChannelRemoteDataStore().createChannel(
            creationType = creationType,
            displayName = option.displayName,
            avatarFileId = option.avatarFileId,
            metadata = option.metadata,
            isPublic = option.isPublic,
            userIds = option.userIds,
            tags = option.tags
        ).flatMap { dto ->
            fetchChannelMarker(dto)
                .andThen(persistAndReturnChannel(dto))
        }
    }

    fun createConversationChannel(
        option: ChannelCreateOption
    ): Single<AmityChannel> {
        return ChannelRemoteDataStore().createConversationChannel(
            displayName = option.displayName,
            avatarFileId = option.avatarFileId,
            metadata = option.metadata,
            userIds = option.userIds,
            tags = option.tags
        ).flatMap { dto ->
            fetchChannelMarker(dto)
                .andThen(persistAndReturnChannel(dto))
        }
    }

    fun joinChannel(channelId: String): Single<AmityChannel> {
        return ChannelRemoteDataStore().joinChannel(channelId)
            .flatMap { dto ->
                fetchChannelMarker(dto)
                    .andThen(persistAndReturnChannel(dto))
            }
            .onErrorResumeNext {
                if (AmityError.USER_IS_BANNED == AmityError.from(it)) {
                    ChannelMembershipRepository().handleMembershipBanned(
                        channelId,
                        AmityCoreClient.getUserId()
                    )
                        .andThen(Single.error(it))
                } else {
                    Single.error(it)
                }
            }
    }

    fun leaveChannel(channelId: String): Completable {
        return ChannelRemoteDataStore().leaveChannel(channelId)
            .flatMapCompletable {
                persistChannels(it)
            }
    }

    fun updateChannel(
        channelId: String,
        option: ChannelUpdateOption
    ): Single<AmityChannel> {
        return ChannelRemoteDataStore().updateChannel(
            channelId = channelId,
            displayName = option.displayName,
            avatarFileId = option.avatarFileId,
            metadata = option.metadata,
            tags = option.tags
        ).flatMap {
            persistAndReturnChannel(it)
        }
    }

    fun getLatestChannel(
        types: Set<AmityChannel.Type>,
        filter: AmityChannelFilter,
        includingTags: AmityTags,
        excludingTags: AmityTags,
        isDeleted: Boolean?,
        dynamicQueryStreamKeyCreator: DynamicQueryStreamKeyCreator,
        nonce: Int
    ) : Flowable<AmityChannel> {
        return ChannelLocalDataStore().getLatestChannel(
            types = types,
            filter = filter,
            includingTags = includingTags,
            excludingTags = excludingTags,
            isDeleted = isDeleted,
            dynamicQueryStreamKeyCreator = dynamicQueryStreamKeyCreator,
            nonce = nonce
        )
            .map {
                ChannelModelMapper().map(it)
            }
    }

    fun getAllJoinedChannels() : Flowable<List<AmityChannel>> {
        return ChannelLocalDataStore().getAllJoinedChannels()
            .map {
                it.map {
                    ChannelModelMapper().map(it)
                }
            }
    }

    internal fun persistChannels(dto: ChannelQueryDto): Completable {
        return ChannelQueryPersister().persist(dto)
    }

    internal fun getChannel(channelId: String): Single<AmityChannel> {
        return ChannelLocalDataStore().observeChannel(channelId)
            .firstOrError()
            .map {
                ChannelModelMapper().map(it)
            }
    }
    
    internal fun getChannels(channelIds: List<String>): Flowable<List<AmityChannel>> {
        val pagerCreator = SinglePagePagerCreator(
            mediator = ChannelListMediator(channelIds = channelIds),
            domainDatasource = ChannelLocalDataStore().getChannels(channelIds),
            modelMapper = ChannelModelMapper()
        )
        
        return pagerCreator.create()
    }

    internal fun persistAndReturnChannel(dto: ChannelQueryDto): Single<AmityChannel> {
        return persistChannels(dto).andThen(getChannel(dto.channelDtoList?.first()?.channelId!!))
    }
    
    internal fun updateMarkerHash(channelId: String, hash: Int) {
        ChannelLocalDataStore().updateMarkerHash(channelId, hash)
    }
    
    private fun fetchChannelMarker(dto: ChannelQueryDto): Completable {
        return if (CoreClient.isUnreadCountEnable()) {
            dto.channelDtoList
                .filter { MarkerSyncEngine.isMarkerSyncSupport(it.type) }
                .map { it.channelId }
                .let { channelIds ->
                    when {
                        channelIds.isEmpty() -> Completable.complete()
                        else -> ChannelMarkerRepository().fetchChannelMarker(channelIds)
                    }
                }
                .onErrorComplete()
        } else {
            Completable.complete()
        }
    }
    
    internal fun updateMessagePreview(channelId: String, messagePreviewId: String): Completable {
        return ChannelLocalDataStore().updateMessagePreview(channelId, messagePreviewId)
    }
    
    internal fun notifyChanges(channelId: String) {
        ChannelLocalDataStore().notifyChanges(channelId)
    }
    
    fun getMessagePreviewId(channelId: String): String? {
        return ChannelLocalDataStore().getMessagePreviewId(channelId)
    }
    
    internal fun notifyChannelsChanges(channelIds: List<String>) {
        ChannelLocalDataStore().notifyChannelsChanges(channelIds)
    }

    fun isChannelCacheExists(channelId: String): Boolean {
        return ChannelLocalDataStore().getChannel(channelId) != null
    }
    
    companion object {
        internal var DEFAULT_PAGE_SIZE = 20
    }
}