package com.amity.socialcloud.sdk.infra.upload

import com.amity.socialcloud.sdk.core.CoreClient
import com.amity.socialcloud.sdk.log.AmityLog
import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityException
import com.amity.socialcloud.sdk.model.core.file.AmityFileAccessType
import com.amity.socialcloud.sdk.socket.model.SocketResponse
import com.amity.socialcloud.sdk.socket.util.EkoGson
import com.ekoapp.core.utils.getCurrentClassAndMethodNames
import com.ekoapp.ekosdk.internal.EkoFileEntity
import com.google.gson.JsonElement
import io.reactivex.rxjava3.core.Flowable
import java.net.ConnectException
import java.net.SocketException
import java.net.SocketTimeoutException
import java.net.UnknownHostException

abstract class AmityUploadService<T : Any> internal constructor() : AmityUploaderParams {

    companion object {
        const val UPLOAD_FILE_PATH = "api/v3/files"
    }

    protected abstract fun makeUploadServiceRequest(): Flowable<T>

    fun transfer(): Flowable<T> {
        return makeUploadServiceRequest()
    }

    internal fun getAccessType(): AmityFileAccessType {
        return CoreClient.getUploadedFileAccessType()
    }

    protected fun parseEkoFileEntity(serverResponse: JsonElement): EkoFileEntity {
        val data: List<EkoFileEntity> =
            EkoGson.get().fromJson(serverResponse, Array<EkoFileEntity>::class.java).toList()
        return data.first()
    }

    protected fun parseErrorResponse(error: Throwable): AmityException {
        return when(error) {
            is UnknownHostException,
            is ConnectException,
            is SocketTimeoutException,
            is SocketException -> {
                AmityException.create(error.message ?: "Connection error", error, AmityError.CONNECTION_ERROR)
            }
            else -> {
                val errorResponse = error.message
                var exception = AmityException.create(error.message ?: "Unknown error", error, AmityError.UNKNOWN)
                try {
                    val errorUploadService =
                            EkoGson.get().fromJson(errorResponse, AmityErrorUploadService::class.java)
                    val response: SocketResponse =
                            EkoGson.get().fromJson(errorUploadService.errorBody, SocketResponse::class.java)
                    exception = AmityException.create(response.message, null, response.code)
                } catch (parsingException: Exception) {
                    val trace = "${getCurrentClassAndMethodNames()}, ${parsingException.message ?: ""}"
                    AmityLog.e(trace)
                    return exception
                }
                exception
            }
        }
    }
}