package com.amity.socialcloud.sdk.api.social.community.update

import com.amity.socialcloud.sdk.model.core.file.AmityImage
import com.amity.socialcloud.sdk.model.social.community.AmityCommunity
import com.amity.socialcloud.sdk.model.social.community.AmityCommunityPostSettings
import com.amity.socialcloud.sdk.model.social.community.AmityCommunityStorySettings
import com.amity.socialcloud.sdk.social.domain.community.UpdateCommunityUseCase
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Single

class AmityCommunityUpdate private constructor(
    private var communityId: String,
    private var displayName: String?,
    private var description: String?,
    private var isPublic: Boolean?,
    private var categoryIds: List<String>?,
    private var metadata: JsonObject?,
    private var avatarFileId: String?,
    private var needApprovalOnPostCreation: Boolean?,
    private var onlyAdminCanPost: Boolean?,
    private var storySettings: AmityCommunityStorySettings? = null
) {

    /**
     * Updates a community
     * @return Async AmityCommunity as Single<AmityCommunity>.
     */
    fun apply(): Single<AmityCommunity> {
        return UpdateCommunityUseCase().execute(
            communityId,
            displayName,
            description,
            categoryIds,
            isPublic,
            metadata,
            avatarFileId,
            needApprovalOnPostCreation,
            onlyAdminCanPost,
            storySettings
        )
    }

    class Builder internal constructor(private val communityId: String) {
        private var displayName: String? = null
        private var description: String? = null
        private var isPublic: Boolean? = null
        private var categoryIds: List<String>? = null
        private var metadata: JsonObject? = null
        private var avatarFileId: String? = null
        private var postSettings: AmityCommunityPostSettings? = null
        private var storySettings: AmityCommunityStorySettings? = null

        /**
         * @param [displayName] the replacing community's displayname value
         * @return the same instance of AmityCommunityUpdate.Builder
         */
        fun displayName(displayName: String): Builder {
            return apply { this.displayName = displayName }
        }

        /**
         * @param [description] the replacing community's description value
         * @return the same instance of AmityCommunityUpdate.Builder
         */
        fun description(description: String): Builder {
            return apply { this.description = description }
        }

        /**
         * @param [isPublic] the replacing community's isPublic value
         * @return the same instance of AmityCommunityUpdate.Builder
         */
        fun isPublic(isPublic: Boolean): Builder {
            return apply { this.isPublic = isPublic }
        }

        /**
         * @param [categoryIds] the replacing community's list of categoryId
         * @return the same instance of AmityCommunityUpdate.Builder
         */
        fun categoryIds(categoryIds: List<String>): Builder {
            return apply { this.categoryIds = categoryIds }
        }

        /**
         * @param [metadata] the replacing community's metadata
         * @return the same instance of AmityCommunityUpdate.Builder
         */
        fun metadata(metadata: JsonObject): Builder {
            return apply { this.metadata = metadata }
        }

        /**
         * @param [avatar] the replacing community's avatar
         * @return the same instance of AmityCommunityUpdate.Builder
         */
        fun avatar(avatar: AmityImage): Builder {
            return apply { this.avatarFileId = avatar.getFileId() }
        }

        /**
         * @param [postSettings] the replacing community's post settings
         * @return the same instance of AmityCommunityUpdate.Builder
         */
        fun postSettings(postSettings: AmityCommunityPostSettings): Builder {
            return apply {
                this.postSettings = postSettings
            }
        }

        fun storySettings(storySettings: AmityCommunityStorySettings): Builder {
            return apply {
                this.storySettings = storySettings
            }
        }

        /**
         * Instantiates AmityCommunityUpdate
         * @return new instance of AmityCommunityUpdate with built params.
         */
        fun build(): AmityCommunityUpdate {
            return AmityCommunityUpdate(
                    communityId = communityId,
                    displayName = displayName,
                    description = description,
                    isPublic = isPublic,
                    categoryIds = categoryIds,
                    metadata = metadata,
                    avatarFileId = avatarFileId,
                    needApprovalOnPostCreation = postSettings?.isPostReviewEnabled,
                    onlyAdminCanPost = postSettings?.onlyAdminCanPost,
                    storySettings = storySettings
            )
        }
    }

}