package com.amity.socialcloud.sdk.social.domain.story.create

import com.amity.socialcloud.sdk.model.core.error.AmityError
import com.amity.socialcloud.sdk.model.core.error.AmityException
import com.amity.socialcloud.sdk.model.core.file.AmityFileInfo
import com.amity.socialcloud.sdk.model.core.file.upload.AmityUploadResult
import com.amity.socialcloud.sdk.model.social.story.AmityStory
import com.amity.socialcloud.sdk.social.data.story.StoryRepository
import com.amity.socialcloud.sdk.social.data.storytarget.StoryTargetRepository
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single

interface CreateStoryUseCase {
    fun <T : AmityFileInfo> createStory(
        targetType: AmityStory.TargetType,
        targetId: String,
        createLocalFile: (uniqueId: String) -> Completable,
        createLocalStory: (uniqueId: String) -> Completable,
        uploadFile: (uniqueId: String) -> Flowable<AmityUploadResult<T>>,
        createRemoteStory: (uniqueId: String, fileId: String) -> Single<String>,
    ): Completable {
        val uniqueId: String = StoryRepository().generateUniqueId()
        return createLocalFile(uniqueId)
            .andThen(createLocalStory(uniqueId))
            .andThen(
                Completable.defer {
                    StoryTargetRepository().updateStoryTargetLocalLastStoryExpiresAt(
                        targetType = targetType,
                        targetId = targetId,
                        localLastStoryExpiresAt = StoryRepository().getStoryNow(storyId = uniqueId)
                            ?.getExpiresAt()
                    )
                }
            )
            .andThen(
                uploadFile(uniqueId)
                    .lastOrError()
                    .flatMapCompletable { uploadResult ->
                        when (uploadResult) {
                            is AmityUploadResult.COMPLETE -> {
                                createRemoteStory(uniqueId, uploadResult.getFile().getFileId())
                                    .flatMapCompletable { storyId ->
                                        Completable.defer {
                                            StoryTargetRepository().updateStoryTargetLastStoryExpiresAt(
                                                targetType = targetType,
                                                targetId = targetId,
                                                lastStoryExpiresAt = StoryRepository().getStoryNow(storyId = storyId)
                                                    ?.getExpiresAt()
                                            )

                                        }
                                    }
                                    .onErrorResumeNext {
                                        val exception =
                                            if (it is AmityException) it else AmityException.create(
                                                "story creation error",
                                                null,
                                                AmityError.UNKNOWN
                                            )
                                        publishError(
                                            targetType = targetType,
                                            targetId = targetId,
                                            uniqueId = uniqueId,
                                            exception = exception
                                        )

                                    }
                            }

                            is AmityUploadResult.ERROR -> {
                                publishError(
                                    targetType = targetType,
                                    targetId = targetId,
                                    uniqueId = uniqueId,
                                    exception = uploadResult.getError()
                                )
                            }

                            else -> {
                                val unknownUploadError = AmityException.create(
                                    "story upload error",
                                    null,
                                    AmityError.UNKNOWN
                                )
                                publishError(
                                    targetType = targetType,
                                    targetId = targetId,
                                    uniqueId = uniqueId,
                                    exception = unknownUploadError
                                )
                            }
                        }
                    }
            )
    }

    private fun publishError(
        targetType: AmityStory.TargetType,
        targetId: String,
        uniqueId: String,
        exception: AmityException
    ): Completable {
        return StoryRepository().updateStorySyncState(uniqueId, AmityStory.State.FAILED)
            .andThen(
                Completable.defer {
                    StoryTargetRepository().updateStoryTargetLocalLastStoryExpiresAt(
                        targetType = targetType,
                        targetId = targetId,
                        localLastStoryExpiresAt = StoryRepository().getStoryNow(storyId = uniqueId)
                            ?.getExpiresAt()
                    )
                }
            )
            .andThen(Completable.error(exception))
    }

}
