package com.amity.socialcloud.sdk.core.presence

import com.amity.socialcloud.sdk.chat.data.channel.ChannelRepository
import com.amity.socialcloud.sdk.chat.data.channel.membership.ChannelMembershipRepository
import com.amity.socialcloud.sdk.core.session.component.SessionComponent
import com.amity.socialcloud.sdk.core.session.eventbus.SessionLifeCycleEventBus
import com.amity.socialcloud.sdk.core.session.eventbus.SessionStateEventBus
import com.amity.socialcloud.sdk.core.session.model.SessionState
import com.amity.socialcloud.sdk.log.AmityLog
import com.amity.socialcloud.sdk.model.chat.channel.AmityChannel
import com.amity.socialcloud.sdk.model.core.error.AmityException
import com.amity.socialcloud.sdk.model.core.presence.AmityChannelPresence
import com.amity.socialcloud.sdk.model.core.presence.AmityUserPresence
import com.ekoapp.ekosdk.internal.data.model.EkoAccount
import com.ekoapp.ekosdk.internal.usecase.presence.GetPresenceNetworkSettingUseCase
import com.ekoapp.ekosdk.internal.usecase.presence.GetUserPresencesUseCase
import com.ekoapp.ekosdk.internal.usecase.presence.SendPresenceHeartbeatUseCase
import io.reactivex.rxjava3.core.BackpressureStrategy
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single
import io.reactivex.rxjava3.disposables.CompositeDisposable
import io.reactivex.rxjava3.schedulers.Schedulers
import io.reactivex.rxjava3.subjects.BehaviorSubject
import org.joda.time.DateTime
import java.util.concurrent.TimeUnit
import java.util.concurrent.atomic.AtomicBoolean

internal class PresenceSyncEngine(
    sessionLifeCycleEventBus: SessionLifeCycleEventBus,
    sessionStateEventBus: SessionStateEventBus
) : SessionComponent(sessionLifeCycleEventBus, sessionStateEventBus) {

    private var config: PresenceConfig? = null
    private var isConfigFetching: Boolean = false
    private var isReadyToSync = AtomicBoolean(false)

    //  HashMap<UserId, Set<ViewId>>
    private val syncedUserIdsMap: HashMap<String, MutableSet<String>> = hashMapOf()
    private val userIdsBuffer: MutableSet<String> = mutableSetOf()
    private val lastFetches: HashMap<String, DateTime?> = hashMapOf()

    //  HashMap<ChannelId, Set<ViewId>>
    private val syncedChannelIdsMap: HashMap<String, MutableSet<String>> = hashMapOf()

    //  HashMap<ChannelId, Set<UserId>>
    private val syncedChannelUserIdsMap: HashMap<String, Set<String>> = hashMapOf()

    private val syncingUserPresencePublisher =
        BehaviorSubject.create<List<AmityUserPresence>>()
    private val syncingChannelPresencePublisher =
        BehaviorSubject.create<List<AmityChannelPresence>>()

    private val heartbeatDisposable = CompositeDisposable()
    private val presenceDisposable = CompositeDisposable()
    private val bufferDisposable = CompositeDisposable()

    fun startHeartbeat(): Completable {
        return getConfigForHeartbeat()
    }

    fun stopHeartbeat() {
        heartbeatDisposable.clear()
    }

    fun syncUserPresence(userId: String, viewId: String) {
        ensureUserSyncLimit(userId, viewId)
        getConfigForPresence()
    }

    fun unsyncUserPresence(userId: String, viewId: String) {
        val viewIds = syncedUserIdsMap[userId] ?: mutableSetOf()
        viewIds.remove(viewId)
        if (viewIds.size == 0) {
            syncedUserIdsMap.remove(userId)
            userIdsBuffer.remove(userId)
        } else {
            syncedUserIdsMap[userId] = viewIds
        }
    }

    fun unsyncAllUserPresence() {
        syncedUserIdsMap.clear()
        stopAllSyncingTimer()
    }

    fun getSyncingUserPresence(): Flowable<List<AmityUserPresence>> {
        return syncingUserPresencePublisher.toFlowable(BackpressureStrategy.BUFFER)
    }

    fun syncChannelPresence(channelId: String, viewId: String) {
        ensureChannelSyncLimit(channelId, viewId)
        getConfigForPresence()
    }

    fun unsyncChannelPresence(channelId: String, viewId: String) {
        val viewIds = syncedChannelIdsMap[channelId] ?: mutableSetOf()
        viewIds.remove(viewId)
        if (viewIds.size == 0) {
            syncedChannelIdsMap.remove(channelId)
            syncedChannelUserIdsMap.remove(channelId)

            getChannelUserIds(channelId)
                .subscribeOn(Schedulers.io())
                .subscribe({ channelUserIds ->
                    userIdsBuffer.removeAll(channelUserIds.toSet())
                }, {

                }).isDisposed
        } else {
            syncedChannelIdsMap[channelId] = viewIds
        }
    }

    fun unsyncAllChannelPresence() {
        syncedChannelIdsMap.clear()
        syncedChannelUserIdsMap.clear()
        stopAllSyncingTimer()
    }

    fun getSyncingChannelPresence(): Flowable<List<AmityChannelPresence>> {
        return syncingChannelPresencePublisher.toFlowable(BackpressureStrategy.BUFFER)
    }

    private fun getConfigForHeartbeat(): Completable {
        return Single.just(config == null)
            .flatMap { isConfigEmpty ->
                if (isConfigEmpty) {
                    GetPresenceNetworkSettingUseCase().execute()
                } else {
                    Single.just(config!!)
                }
            }
            .retryWhen {
                it.delay(
                    PresenceConfig.SYNC_CONFIG_RETRY_INTERVAL,
                    TimeUnit.MILLISECONDS
                )
            }
            .map {
                config = it
                config?.isSyncEnabled() ?: false
            }.flatMapCompletable { isSyncEnabled ->
                if (isSyncEnabled) {
                    createHeartbeatTimer()
                    Completable.complete()
                } else {
                    Completable.error(
                        AmityException.create(
                            "Presence sync is not enabled for this network",
                            null,
                            400300
                        )
                    )
                }
            }
    }

    private fun createHeartbeatTimer() {
        heartbeatDisposable.clear()

        config?.getHeartbeatSyncInterval()?.let { heartbeatSyncInterval ->
            Flowable.interval(
                0,
                heartbeatSyncInterval,
                TimeUnit.MILLISECONDS
            )
                .map { sendHeartbeat() }
                .subscribeOn(Schedulers.io())
                .subscribe()
                .let(heartbeatDisposable::add)
        }
    }

    private fun sendHeartbeat() {
        if (!isReadyToSync.get()) return

        SendPresenceHeartbeatUseCase().execute()
            .subscribeOn(Schedulers.io())
            .subscribe()
    }

    private fun getConfigForPresence() {
        if (!shouldConfigFetchForPresence()) return

        GetPresenceNetworkSettingUseCase().execute()
            .subscribeOn(Schedulers.io())
            .doOnSubscribe { isConfigFetching = true }
            .subscribe({ presenceConfig ->
                config = presenceConfig
                isConfigFetching = false
                createPresenceSyncTimer()
            }, {
                isConfigFetching = false
            }).isDisposed
    }

    private fun shouldConfigFetchForPresence(): Boolean {
        return if (isConfigFetching) {
            //  config is fetching, prevent duplicate fetching
            false
        } else if (config != null) {
            //  config already fetched
            if (presenceDisposable.size() > 0
                && bufferDisposable.size() > 0
            ) {
                //  presence and buffer timer already started
                false
            } else {
                //  presence and buffer disposable are empty, assume timer are not running
                createPresenceSyncTimer()
                false
            }
        } else {
            //  config is empty
            true
        }
    }

    private fun createPresenceSyncTimer() {
        if (config?.isSyncEnabled() == true) {
            createPresenceTimer()
            createBufferTimer()
        } else {
            AmityLog.tag(TAG).e("Presence sync is not enabled for this network")
            syncingUserPresencePublisher.onNext(emptyList())
            syncingChannelPresencePublisher.onNext(emptyList())
        }
    }

    private fun createPresenceTimer() {
        presenceDisposable.clear()

        config?.getPresenceSyncInterval()?.let { presenceSyncInterval ->
            Flowable.interval(
                INITIAL_DELAY,
                presenceSyncInterval,
                TimeUnit.MILLISECONDS
            )
                .subscribeOn(Schedulers.io())
                .map {
                    val userIds: MutableSet<String> = syncedUserIdsMap.keys.toMutableSet()
                    syncedChannelUserIdsMap.values.map { channelUserIds ->
                        userIds.addAll(channelUserIds)
                    }
                    fetchUserPresence(userIds.toList())
                }
                .subscribe()
                .let(presenceDisposable::add)
        }
    }

    private fun createBufferTimer() {
        bufferDisposable.clear()

        Flowable.interval(
            INITIAL_DELAY,
            1000,
            TimeUnit.MILLISECONDS
        )
            .subscribeOn(Schedulers.io())
            .map {
                fetchBufferUserPresence()
            }
            .subscribe()
            .let(bufferDisposable::add)
    }

    private fun fetchBufferUserPresence() {
        if (userIdsBuffer.isEmpty()) return
        userIdsBuffer.filter { userIdBuffer ->
            val lastFetch = lastFetches[userIdBuffer]
            val isExpired = lastFetch?.plusSeconds(30)?.isBeforeNow ?: true

            lastFetch == null || isExpired
        }.let { filteredUserIds ->
            fetchUserPresence(filteredUserIds)
            userIdsBuffer.clear()
        }
    }

    private fun fetchUserPresence(userIds: List<String>) {
        if (userIds.isEmpty() || !isReadyToSync.get()) return

        GetUserPresencesUseCase().execute(userIds.toList())
            .subscribeOn(Schedulers.io())
            .subscribe({ presences ->
                emitPresencesToObserver(presences)
                presences.map { presence ->
                    lastFetches[presence.getUserId()] = DateTime.now()
                }
            }, {
                AmityLog.tag(TAG).e("Error in fetching user presence: $it")
                syncingUserPresencePublisher.onNext(emptyList())
                syncingChannelPresencePublisher.onNext(emptyList())
            }).isDisposed
    }

    private fun emitPresencesToObserver(presences: List<AmityUserPresence>) {
        //  compose user presences list
        syncedUserIdsMap.keys
            .mapNotNull { userId ->
                presences.find { it.getUserId() == userId }
            }.let { userPresences ->
                if (userPresences.isNotEmpty()) {
                    syncingUserPresencePublisher.onNext(userPresences)
                }
            }

        //  compose channel presences list
        syncedChannelUserIdsMap
            .mapNotNull { (channelId, userIds) ->
                val userPresences = userIds.mapNotNull { userId ->
                    presences.find { it.getUserId() == userId }
                }
                if (userPresences.isNotEmpty()) {
                    AmityChannelPresence(
                        channelId = channelId,
                        userPresences = userPresences
                    )
                } else {
                    null
                }
            }.let { channelPresences ->
                if (channelPresences.isNotEmpty()) {
                    syncingChannelPresencePublisher.onNext(channelPresences)
                }
            }
    }

    private fun ensureUserSyncLimit(userId: String, viewId: String) {
        if (syncedUserIdsMap.size >= 20 && !syncedUserIdsMap.containsKey(userId)) {
            AmityLog.tag(TAG).e("Maximum limit reached for syncing users")
        } else {
            val viewIds = syncedUserIdsMap[userId] ?: mutableSetOf()
            viewIds.add(viewId)
            syncedUserIdsMap[userId] = viewIds
            userIdsBuffer.add(userId)
        }
    }

    private fun ensureChannelSyncLimit(channelId: String, viewId: String) {
        if (syncedChannelIdsMap.size >= 20 && !syncedChannelIdsMap.containsKey(channelId)) {
            AmityLog.tag(TAG).e("Maximum limit reached for syncing channels")
        } else {
            checkChannelType(channelId)
                .subscribeOn(Schedulers.io())
                .subscribe({ isConversationChannel ->
                    if (isConversationChannel) {
                        val viewIds = syncedChannelIdsMap[channelId] ?: mutableSetOf()
                        viewIds.add(viewId)
                        syncedChannelIdsMap[channelId] = viewIds

                        getChannelUserIds(channelId)
                            .subscribeOn(Schedulers.io())
                            .map { channelUserIds ->
                                syncedChannelUserIdsMap[channelId] = channelUserIds.toSet()
                                userIdsBuffer.addAll(channelUserIds)
                            }.subscribe()
                    } else {
                        AmityLog.tag(TAG)
                            .e("Conversation channel with given <$channelId> not found.")
                    }
                }, {

                }).isDisposed
        }
    }

    private fun checkChannelType(channelId: String): Single<Boolean> {
        return ChannelRepository().getChannel(channelId)
            .subscribeOn(Schedulers.io())
            .map {
                it.getChannelType() == AmityChannel.Type.CONVERSATION
            }
    }

    private fun getChannelUserIds(channelId: String): Single<List<String>> {
        return ChannelMembershipRepository().getChannelMembers(channelId)
            .map { memberships ->
                memberships.map { it.getUserId() }
            }
    }

    private fun stopAllSyncingTimer() {
        //  both syncing user and channel ids are empty, stop the timer
        if (syncedUserIdsMap.isEmpty() && syncedChannelIdsMap.isEmpty()) {
            userIdsBuffer.clear()

            presenceDisposable.clear()
            bufferDisposable.clear()
        }
    }

    override fun onSessionStateChange(sessionState: SessionState) {
        when (sessionState) {
            SessionState.Established -> isReadyToSync.set(true)
            else -> isReadyToSync.set(false)
        }
    }

    override fun establish(account: EkoAccount) {
        isReadyToSync.set(true)
    }

    override fun destroy() {
        isReadyToSync.set(false)
        stopHeartbeat()
        unsyncAllUserPresence()
        unsyncAllChannelPresence()
    }

    override fun handleTokenExpire() {
        isReadyToSync.set(false)
    }

    companion object {
        private const val TAG = "AmityPresenceState"
        private const val INITIAL_DELAY = 500L
    }
}