package com.amity.socialcloud.sdk.core.data.follow

import androidx.paging.PagingSource
import com.amity.socialcloud.sdk.model.core.follow.AmityFollowStatus
import com.amity.socialcloud.sdk.model.core.follow.AmityFollowStatusFilter
import com.ekoapp.ekosdk.internal.data.UserDatabase
import com.ekoapp.ekosdk.internal.data.model.EkoFollowCountEntity
import com.ekoapp.ekosdk.internal.data.model.EkoUserFollowStatusEntity
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable

internal class FollowLocalDataStore {

    fun saveFollow(follows: List<EkoUserFollowStatusEntity>): Completable {
        return Completable.fromAction {
            UserDatabase.get().followDao().save(follows)
        }
    }

    fun saveFollowCount(followCounts: List<EkoFollowCountEntity>): Completable {
        return Completable.fromAction {
            UserDatabase.get().followCountDao().save(followCounts)
        }
    }

    fun getFollowCount(userId: String): EkoFollowCountEntity? {
        return UserDatabase.get().followCountDao().getByIdNow(userId)
    }

    fun observeFollowCount(userId: String): Flowable<EkoFollowCountEntity> {
        return UserDatabase.get().followCountDao().getById(userId)
    }

    fun observeUserFollow(sourceUserId: String, targetUserId: String): Flowable<EkoUserFollowStatusEntity> {
        return UserDatabase.get().followDao().getUserFollowById(sourceUserId, targetUserId)
    }

    fun getFollowingsPagingSource(
        userId: String,
        filter: AmityFollowStatusFilter
    ): PagingSource<Int, EkoUserFollowStatusEntity> {
        return UserDatabase.get().followingPagingDao().getFollowingPagingSource(
            userId = userId,
            followStatusFilter = filter
        )
    }

    fun getFollowersPagingSource(
        userId: String,
        filter: AmityFollowStatusFilter
    ): PagingSource<Int, EkoUserFollowStatusEntity> {
        return UserDatabase.get().followingPagingDao().getFollowerPagingSource(
            userId = userId,
            followStatusFilter = filter
        )
    }

    fun getLatestFollowing(
        userId: String,
        filter: AmityFollowStatusFilter
    ): Flowable<EkoUserFollowStatusEntity> {
        return UserDatabase.get().followDao().getLatestFollowing(userId, filter.statuses)
    }

    fun getLatestFollower(
        userId: String,
        filter: AmityFollowStatusFilter
    ): Flowable<EkoUserFollowStatusEntity> {
        return UserDatabase.get().followDao().getLatestFollower(userId, filter.statuses)
    }

    fun updateFollower(
        userId: String,
        status: AmityFollowStatus
    ) {
        return UserDatabase.get().followDao().updateFollower(userId, status.apiKey)
    }

    fun updateFollowing(
        userId: String,
        status: AmityFollowStatus
    ) {
        return UserDatabase.get().followDao().updateFollowing(userId, status.apiKey)
    }

    fun getFollowCountByIdNow(userId: String): EkoFollowCountEntity? {
        return UserDatabase.get().followCountDao().getByIdNow(userId)
    }

    fun updateFollowerCount(userId: String, followerCount: Int) {
        UserDatabase.get().followCountDao().updateFollowerCount(userId, followerCount)
    }

    fun updatePendingCount(userId: String, followerCount: Int) {
        UserDatabase.get().followCountDao().updatePendingCount(userId, followerCount)
    }

    fun updateFollowingCount(userId: String, followerCount: Int) {
        UserDatabase.get().followCountDao().updateFollowingCount(userId, followerCount)
    }

    fun getBlockedPagingSource(userId: String): PagingSource<Int, EkoUserFollowStatusEntity> {
        return UserDatabase.get().followingPagingDao().getBlockedPagingSource(userId)
    }
}