package com.amity.socialcloud.sdk.api.social.post.update

import com.amity.socialcloud.sdk.model.core.file.AmityFile
import com.amity.socialcloud.sdk.model.core.file.AmityImage
import com.amity.socialcloud.sdk.model.core.file.AmityVideo
import com.amity.socialcloud.sdk.model.social.post.AmityPost
import com.amity.socialcloud.sdk.model.social.post.AmityPostAttachment
import com.amity.socialcloud.sdk.social.domain.post.update.PostEditUseCase
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Completable


class AmityPostUpdate internal constructor(
    private val postId: String,
    private val text: String?,
    private val metadata: JsonObject?,
    private val mentionees: List<String>?,
    private val attachments: List<AmityPostAttachment>?
)  {

    private fun getData(): JsonObject {
        val data = JsonObject()
        text?.let {
            data.addProperty("text", text)
        }
        return data
    }

    /**
     * Updates a post
     * @return Async post as Single<AmityPost>
     */
    fun apply(): Completable {
        return PostEditUseCase().execute(postId, getData(), metadata, mentionees, attachments)
    }

    class Builder internal constructor(private val postId: String) {

        private var text: String? = null
        private var metadata: JsonObject? = null
        private var mentionees: List<String>? = null
        private var attachments: List<AmityPostAttachment>? = null

        /**
         * @param [text] to replace text on the post
         * @return the same instance of AmityPostUpdate.Builder
         */
        fun text(text: String): Builder {
            return apply {
                this.text = text
            }
        }

        /**
         * @param [attachments] to replace video children
         * @return the same instance of AmityPostUpdate.Builder
         */
        fun attachments(vararg attachments: AmityVideo): Builder {
            this.attachments = attachments.toList().map { AmityPostAttachment(it.getFileId(),  AmityPost.DataType.VIDEO.getApiKey()) }
            return this
        }

        /**
         * @param [attachments] to replace image children
         * @return the same instance of AmityPostUpdate.Builder
         */
        fun attachments(vararg attachments: AmityImage): Builder {
            this.attachments = attachments.toList().map { AmityPostAttachment(it.getFileId(),  AmityPost.DataType.IMAGE.getApiKey()) }
            return this
        }

        /**
         * @param [attachments] to replace file children
         * @return the same instance of AmityPostUpdate.Builder
         */
        fun attachments(vararg attachments: AmityFile): Builder {
            this.attachments = attachments.toList().map { AmityPostAttachment(it.getFileId(),  AmityPost.DataType.FILE.getApiKey()) }
            return this
        }

        /**
         * @param [metadata] to replace metadata
         * @return the same instance of AmityPostUpdate.Builder
         */
        fun metadata(metadata: JsonObject): Builder {
            this.metadata = metadata
            return this
        }

        /**
         * @param [userIds] to update the list of userId of users to be mentioned by this post
         * @return the same instance of AmityPostUpdate.Builder
         */
        fun mentionUsers(userIds: List<String>): Builder {
            this.mentionees = userIds
            return this
        }

        /**
         * Instantiates AmityPostUpdate with built params
         * @return new instance of AmityPostUpdate
         */
        fun build(): AmityPostUpdate {
            return AmityPostUpdate(postId, text, metadata, mentionees, attachments)
        }

    }

}