package com.amity.socialcloud.sdk.chat.data.channel.membership

import com.amity.socialcloud.sdk.infra.retrofit.request.QueryOptionsRequestParams
import com.ekoapp.ekosdk.internal.api.AmityHttpClient
import com.ekoapp.ekosdk.internal.api.dto.ChannelMembershipQueryDto
import com.ekoapp.ekosdk.internal.api.socket.request.*
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Single
import org.joda.time.Duration

class ChannelMembershipRemoteDataStore {

    fun queryChannelMembers(
        channelId: String,
        keyword: String? = null,
        roles: List<String>? = null,
        memberships: List<String>? = null,
        sortBy: String? = null,
        options: QueryOptionsRequestParams
    ): Single<ChannelMembershipQueryDto> {
        return AmityHttpClient.get(ChannelMembershipApi::class).flatMap {
            it.queryChannelUsers(
                channelId = channelId,
                keyword = keyword,
                roles = roles?.takeIf { it.isNotEmpty() },
                memberships = memberships?.takeIf { it.isNotEmpty() },
                sortBy = sortBy,
                limit = options.limit,
                token = options.token
            )
        }
    }

    fun addChannelMembers(
        channelId: String,
        userIds: List<String>
    ): Single<ChannelMembershipQueryDto> {
        return AmityHttpClient.get(ChannelMembershipApi::class).flatMap {
            it.addChannelUsers(channelId, ChannelAddUsersRequest(channelId, userIds))
        }
    }

    fun removeChannelMembers(
        channelId: String,
        userIds: List<String>
    ): Single<ChannelMembershipQueryDto> {
        return AmityHttpClient.get(ChannelMembershipApi::class).flatMap {
            it.deleteChannelUsers(channelId, ChannelRemoveUsersRequest(channelId, userIds))
        }
    }

    fun assignChannelRole(channelId: String, role: String, userIds: List<String>): Single<ChannelMembershipQueryDto> {
        val request = ChannelAddRoleRequest(channelId, role, userIds)
        return AmityHttpClient.get(ChannelMembershipApi::class).flatMap {
            it.assignChannelRole(channelId, request)
        }
    }

    fun unassignChannelRole(channelId: String, role: String, userIds: List<String>): Single<ChannelMembershipQueryDto> {
        val request = ChannelRemoveRoleRequest(channelId, role, userIds)
        return AmityHttpClient.get(ChannelMembershipApi::class).flatMap {
            it.unassignChannelRole(channelId, request)
        }
    }

    fun banChannelMembers(channelId: String, userIds: List<String>): Single<ChannelMembershipQueryDto> {
        val request = ChannelBanUserRequest(userIds)
        return AmityHttpClient.get(ChannelMembershipApi::class).flatMap {
            it.banChannelUsers(channelId, request)
        }
    }

    fun unbanChannelMembers(channelId: String, userIds: List<String>): Single<ChannelMembershipQueryDto> {
        val request = ChannelUnbanUserRequest(channelId, userIds)
        return AmityHttpClient.get(ChannelMembershipApi::class).flatMap {
            it.unbanChannelUsers(channelId, request)
        }
    }

    fun muteChannel(channelId: String, timeout: Duration): Completable {
        val request = ChannelSetMutedRequest(channelId, timeout.millis)
        return AmityHttpClient.get(ChannelMembershipApi::class).flatMapCompletable {
            it.muteChannel(channelId, request)
        }
    }

    fun muteChannelMembers(channelId: String, timeout: Duration, userIds: List<String>): Completable {
        val request = ChannelSetMutedUsersRequest(channelId, timeout.millis, userIds)
        return AmityHttpClient.get(ChannelMembershipApi::class).flatMapCompletable {
            it.muteChannelUser(channelId, request)
        }
    }

}