package com.amity.socialcloud.sdk.api.core

import com.amity.socialcloud.sdk.api.core.ads.AmityAdRepository
import com.amity.socialcloud.sdk.api.core.encryption.AmityDBEncryption
import com.amity.socialcloud.sdk.api.core.endpoint.AmityEndpoint
import com.amity.socialcloud.sdk.api.core.events.AmityTopicSubscription
import com.amity.socialcloud.sdk.api.core.file.AmityFileRepository
import com.amity.socialcloud.sdk.api.core.notification.AmityNotification
import com.amity.socialcloud.sdk.api.core.permission.AmityPermissionValidator
import com.amity.socialcloud.sdk.api.core.presence.AmityPresenceService
import com.amity.socialcloud.sdk.api.core.presence.AmityUserPresenceRepository
import com.amity.socialcloud.sdk.api.core.reaction.AmityReactionRepository
import com.amity.socialcloud.sdk.api.core.session.AmitySessionEstablisher
import com.amity.socialcloud.sdk.api.core.user.AmityUserRepository
import com.amity.socialcloud.sdk.api.core.user.notification.AmityUserNotification
import com.amity.socialcloud.sdk.api.core.user.update.AmityUserUpdate
import com.amity.socialcloud.sdk.core.CoreClient
import com.amity.socialcloud.sdk.core.domain.networksettings.ValidateTextsUseCase
import com.amity.socialcloud.sdk.core.domain.networksettings.ValidateUrlsUseCase
import com.amity.socialcloud.sdk.core.domain.notification.RegisterDeviceForNotificationUseCase
import com.amity.socialcloud.sdk.core.domain.notification.UnregisterAllUsersForNotificationUseCase
import com.amity.socialcloud.sdk.core.domain.session.ActiveUserIdGetUseCase
import com.amity.socialcloud.sdk.core.domain.session.LogoutUseCase
import com.amity.socialcloud.sdk.core.domain.session.SecureLogoutUseCase
import com.amity.socialcloud.sdk.core.session.model.SessionState
import com.amity.socialcloud.sdk.model.core.content.AmityContentCheck
import com.amity.socialcloud.sdk.model.core.events.AmityTopic
import com.amity.socialcloud.sdk.model.core.permission.AmityPermission
import com.amity.socialcloud.sdk.model.core.session.AmityGlobalBanEvent
import com.amity.socialcloud.sdk.model.core.session.SessionHandler
import com.amity.socialcloud.sdk.model.core.unread.UserUnread
import com.amity.socialcloud.sdk.model.core.user.AmityUser
import com.amity.socialcloud.sdk.streamapi.AmityClientConfiguration
import com.amity.socialcloud.sdk.video.data.stream.StreamFunction
import com.ekoapp.ekosdk.internal.api.EkoSocket
import com.ekoapp.ekosdk.internal.usecase.content.GetContentCheckUseCase
import com.ekoapp.ekosdk.sdk.BuildConfig
import io.reactivex.rxjava3.core.Completable
import io.reactivex.rxjava3.core.Flowable
import io.reactivex.rxjava3.core.Single

object AmityCoreClient {

    /* begin_public_function
	  id: client.read_session_state
	*/
    fun getCurrentSessionState(): SessionState {
        return CoreClient.currentSessionState
    }
    /* end_public_function */

    /**
     * This method setup the SDK by overriding default endpoints
     *
     * @param apiKey an api key to be used with the SDK
     * @param endpoint an endpoint model to override default endpoints
     *
     */
    fun setup(
        apiKey: String,
        endpoint: AmityEndpoint = AmityEndpoint.SG,
        dbEncryption: AmityDBEncryption = AmityDBEncryption.NONE
    ): Completable {
        return CoreClient.setup(apiKey, endpoint, dbEncryption)
    }

    @Deprecated("Use registerPushNotification() instead")
    fun registerDeviceForPushNotification(): Completable {
        return registerPushNotification()
    }

    @Deprecated("Use unregisterPushNotification() instead")
    fun unregisterDeviceForPushNotification(): Completable {
        return unregisterPushNotification()
    }

    /* begin_public_function
	  id: client.register_push
	*/
    fun registerPushNotification(): Completable {
        return RegisterDeviceForNotificationUseCase().execute()
    }
    /* end_public_function */

    /* begin_public_function
	  id: client.unregister_push
	*/
    fun unregisterPushNotification(): Completable {
        return UnregisterAllUsersForNotificationUseCase().execute()
    }
    /* end_public_function */

    /* begin_public_function
	  id: client.disconnect
	*/
    fun disconnect(): Completable {
        return CoreClient.disconnect()
    }
    /* end_public_function */

    /* begin_public_function
	  id: client.login
	  api_style: async
	*/
    fun login(userId: String, sessionHandler: SessionHandler?): AmitySessionEstablisher.Builder {
        return CoreClient.login(userId, sessionHandler)
    }
    /* end_public_function */

    /* begin_public_function
	  id: client.logout
	*/
    fun logout(): Completable {
        return LogoutUseCase().execute()
    }
    /* end_public_function */

    /* begin_public_function
	  id: client.secure_logout
	*/
    fun secureLogout(): Completable {
        return SecureLogoutUseCase().execute()
    }
    /* end_public_function */

    /* begin_public_function
	  id: client.get_current_user
	*/
    fun getUserId(): String {
        return ActiveUserIdGetUseCase().execute()
    }
    /* end_public_function */

    /* begin_public_function
	  id: client.update_user
	  api_style: async
	*/
    fun editUser(): AmityUserUpdate.Builder {
        return AmityUserUpdate.Builder(getUserId())
    }
    /* end_public_function */

    @Deprecated("Use editUser() instead")
    fun updateUser(): AmityUserUpdate.Builder {
        return editUser()
    }

    fun hasPermission(permission: AmityPermission): AmityPermissionValidator {
        return AmityPermissionValidator(permission)
    }

    fun getCurrentUser(): Flowable<AmityUser> {
        return newUserRepository().getCurrentUser()
    }

    fun newUserRepository(): AmityUserRepository {
        return AmityUserRepository()
    }

    fun newAdRepository(): AmityAdRepository {
        return AmityAdRepository()
    }

    fun newFileRepository(): AmityFileRepository {
        return AmityFileRepository()
    }

    fun newReactionRepository(): AmityReactionRepository {
        return AmityReactionRepository()
    }

    fun getAmityCoreSdkVersion(): String {
        return BuildConfig.VERSION_NAME
    }

    fun getGlobalBanEvents(): Flowable<AmityGlobalBanEvent> {
        return EkoSocket.globalBanEvents
    }

    @Deprecated("Use notifications.user() instead")
    fun notification(): AmityUserNotification {
        return AmityUserNotification()
    }

    /* begin_public_function
	  id: client.notifications
	*/
    fun notifications(): AmityNotification {
        return AmityNotification()
    }
    /* end_public_function */

    fun getConfiguration(): AmityClientConfiguration {
        return AmityClientConfiguration(StreamFunction())
    }

    fun subscription(topic: AmityTopic): AmityTopicSubscription {
        return AmityTopicSubscription(topic)
    }

    fun getContentCheck(): Single<AmityContentCheck> {
        return GetContentCheckUseCase().execute()
    }

    /* begin_public_function
	  id: client.observe_session_state
	  api_style: value_observation
	*/
    fun observeSessionState(): Flowable<SessionState> {
        return CoreClient.observeSessionState()
    }
    /* end_public_function */
    
    @Deprecated("Using enableUnreadCount() instead", ReplaceWith("AmityCoreClient.enableUnreadCount()", "com.amity.socialcloud.sdk.api.core.AmityCoreClient"))
    fun startUnreadCountSyncing() {
        CoreClient.startMarkerSyncing()
    }
    
    
    @Deprecated("Using enableUnreadCount() instead", ReplaceWith("AmityCoreClient.enableUnreadCount()", "com.amity.socialcloud.sdk.api.core.AmityCoreClient"))
    fun startUnreadSync() {
        CoreClient.startMarkerSyncing()
    }

    @Deprecated("Remove this if changing to use enableUnreadCount()")
    fun stopUnreadCountSyncing() {
        CoreClient.stopMarkerSyncing()
    }
    
    @Deprecated("Remove this if changing to use enableUnreadCount()")
    fun stopUnreadSync() {
        CoreClient.stopMarkerSyncing()
    }

    @Deprecated("Use observeUserUnread() instead")
    fun getTotalUnreadCount(): Flowable<Int> {
        return CoreClient.observeUserUnreadCount()
    }

    fun observeUserUnread(): Flowable<UserUnread> {
        return CoreClient.observeUserUnread()
    }

    fun presence(): AmityPresenceService {
        return CoreClient.presence()
    }

    fun newUserPresenceRepository(): AmityUserPresenceRepository {
        return CoreClient.newUserPresenceRepository()
    }
    
    fun enableUnreadCount() {
        CoreClient.enableUnreadCount()
    }

    fun validateUrls(urls: List<String>) : Completable {
        return ValidateUrlsUseCase().execute(urls)
    }

    fun validateTexts(texts: List<String>) : Completable {
        return ValidateTextsUseCase().execute(texts)
    }

}