package com.amity.socialcloud.sdk.social.data.post

import com.amity.socialcloud.sdk.model.core.mention.AmityMentioneeTarget
import com.amity.socialcloud.sdk.model.social.post.AmityPost
import com.amity.socialcloud.sdk.model.social.post.AmityPostAttachment
import com.ekoapp.ekosdk.internal.api.AmityHttpClient
import com.ekoapp.ekosdk.internal.api.dto.EkoDeletionDto
import com.ekoapp.ekosdk.internal.api.dto.EkoPostQueryDto
import com.ekoapp.ekosdk.internal.api.socket.request.CreatePostV4Request
import com.ekoapp.ekosdk.internal.api.socket.request.PostUpdateRequest
import com.google.gson.JsonObject
import io.reactivex.rxjava3.core.Single

internal class PostRemoteDataStore {

    fun fetchPost(postId: String): Single<EkoPostQueryDto> {
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.getPost(postId)
        }
    }

    fun approvePost(postId: String): Single<EkoPostQueryDto> {
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.approvePost(postId)
        }
    }

    fun declinePost(postId: String): Single<EkoPostQueryDto> {
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.declinePost(postId)
        }
    }

    fun deletePost(postId: String, hardDelete: Boolean): Single<EkoDeletionDto> {
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.deletePost(postId = postId, permanent = hardDelete)
        }
    }

    fun createPost(
        targetType: String, targetId: String,
        data: JsonObject, attachments: List<AmityPostAttachment>?,
        dataType: AmityPost.DataType? = null,
        metadata: JsonObject?,
        mentionees: List<AmityMentioneeTarget>?
    ): Single<EkoPostQueryDto> {
        val createPostRequest = CreatePostV4Request(
            targetType,
            targetId,
            data,
            dataType?.getApiKey(),
            attachments,
            metadata,
            mentionees
        )
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.createPost(createPostRequest)
        }
    }

    fun flagPost(postId: String): Single<EkoPostQueryDto> {
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.flagPost(postId)
        }
    }

    fun unFlagPost(postId: String): Single<EkoPostQueryDto> {
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.unflagPost(postId)
        }
    }

    fun editPost(
        postId: String,
        data: JsonObject,
        metadata: JsonObject?,
        mentionees: List<AmityMentioneeTarget>?,
        attachments: List<AmityPostAttachment>?
    ): Single<EkoPostQueryDto> {
        val postUpdateRequest = PostUpdateRequest(postId, data, metadata, mentionees, attachments)
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.updatePost(postId, postUpdateRequest)
        }
    }

    fun getCommunityPosts(
        targetId: String,
        sortBy: String,
        isDeleted: Boolean?,
        feedType: String?,
        dataTypes: List<String>? = null,
        matchingOnlyParentPost: Boolean = true,
        limit: Int? = null,
        token: String? = null
    ): Single<EkoPostQueryDto> {
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.queryPosts(
                targetType = "community",
                targetId = targetId,
                isDeleted = isDeleted,
                feedType = feedType?.let { listOf(it) },
                dataTypes = dataTypes,
                sortBy = sortBy,
                matchingOnlyParentPost = matchingOnlyParentPost,
                limit = limit,
                token = token
            )
        }
    }

    fun getUserPosts(
        userId: String,
        isDeleted: Boolean?,
        postTypes: List<String>?,
        sortBy: String,
        matchingOnlyParentPost: Boolean = true,
        limit: Int? = null,
        token: String? = null
    ): Single<EkoPostQueryDto> {
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.queryPosts(
                targetType = "user",
                targetId = userId,
                isDeleted = isDeleted,
                dataTypes = postTypes,
                sortBy = sortBy,
                matchingOnlyParentPost = matchingOnlyParentPost,
                limit = limit,
                token = token
            )
        }

    }

    fun getGlobalFeedPosts(limit: Int? = null, token: String? = null): Single<EkoPostQueryDto> {
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.globalFeedV4(limit = limit, token = token)
        }
    }

    fun getCustomRankingPosts(limit: Int? = null, token: String? = null): Single<EkoPostQueryDto> {
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.globalFeedV5(limit = limit, token = token)
        }
    }

    fun getPostByIds(postIds: List<String>) : Single<EkoPostQueryDto> {
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.getPostByIds(postIds)
        }
    }

    fun isFlaggedByMe(postId: String): Single<JsonObject> {
        return AmityHttpClient.get(PostApi::class).flatMap {
            it.isFlaggedByMe(postId)
        }
    }

}