package com.flybits.android.push.api

import android.content.Context
import com.flybits.android.push.PushScope
import com.flybits.android.push.db.PushDatabase
import com.flybits.android.push.db.caching.PushCacheLoader
import com.flybits.android.push.db.converters.PushDataToPushConverter
import com.flybits.android.push.deserializations.DeserializePushNotificationV2
import com.flybits.android.push.models.PushData
import com.flybits.android.push.models.newPush.Push
import com.flybits.android.push.utils.PushQueryParameters
import com.flybits.commons.library.api.FlyAway
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback
import com.flybits.commons.library.deserializations.DeserializePagedResponse
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.http.RequestStatus
import com.flybits.internal.db.CommonsDatabase
import com.flybits.internal.db.models.CachingEntry
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.launch
import java.util.*

/**
 * The [PushAPIManager] is responsible for all CRUD operations on the [Push] notifications entity
 * such as retrieving [Push] notifications through the [PushAPIManager.get] method based on filters
 * defined in the [PushQueryParameters] and deleting a specific notification based on the [Push.id].
 */
object PushAPIManager {

    internal val API = PushScope.ROOT + "/notifications"

    /**
     * Delete the Push Notification that is uniquely identified through the [id].
     *
     * @param context The context of the application.
     * @param callback The call that indicated whether or not the request was unsuccessful.
     *
     * @return The [Job] of the request being made to Flybits.
     */
    @JvmStatic
    fun delete(id : String, context : Context, callback : BasicResultCallback? = null) : Job {

        return CoroutineScope((Dispatchers.IO)).launch {
            val removeAssociationResult = FlyAway.delete(context,
                API,
                "Push.delete",
                id)

            if (removeAssociationResult.status == RequestStatus.COMPLETED){
                PushDatabase.getDatabase(context).pushDataDAO().deleteById(id)
                CommonsDatabase.getDatabase(context).cachingEntryDAO()
                    .deleteByCachingKeyAndValue(PushCacheLoader.PUSH_CACHE_KEY, id)

            }

            CoroutineScope(Dispatchers.Main).launch {
                when (removeAssociationResult.status) {
                    //If it can't be found it means it's not there so no need to remove
                    RequestStatus.NOT_FOUND, RequestStatus.COMPLETED -> callback?.onSuccess()
                    else-> callback?.onException(removeAssociationResult.exception)
                }
            }
        }
    }

    /**
     * Get the Push Notifications for the connected user.
     *
     * @param context The context of the application.
     * @param params The [PushQueryParameters] that is used for creating filters on the get request
     * to narrow down the list of returned object based on specific query.
     * @param callback The call that indicated whether or not the request was unsuccessful.
     *
     * @return The [Job] of the request being made to Flybits.
     */
    @JvmStatic
    fun get(context: Context, params : PushQueryParameters, callback : PagedResultCallback<Push>?) : Job{

        return CoroutineScope((Dispatchers.IO)).launch {

            try {
                val singleObjectDeserializer = DeserializePushNotificationV2()
                val result = FlyAway.get<PushData>(
                    context,
                    API,
                    params,
                    DeserializePagedResponse<PushData>(singleObjectDeserializer),
                    "Push.get"
                )

                when (result.status) {

                    RequestStatus.COMPLETED -> {

                        var listOfIds  = emptyList<String>()
                        //Handle Caching for Push Notifications
                        if ((params.queryParams["offset"].isNullOrEmpty()
                                    || params.queryParams["offset"]?.get(0) == "0")
                                    && params.cachingKey != null
                        ) {
                            listOfIds = CommonsDatabase.getDatabase(context).cachingEntryDAO()
                                .getUniqueIdsByCachingKey(params.cachingKey)
                        }

                        val maxNumberOfRecordsToSave =
                            if (params.cachingLimit < result.result.items.size + result.result.pagination.offset)
                                params.cachingLimit
                            else
                                result.result.items.size

                        if (params.cachingKey != null && maxNumberOfRecordsToSave > 0) {
                            val entries = ArrayList<CachingEntry>()
                            val pushesToCache = ArrayList<PushData>()
                            for (i in 0 until result.result.items.size) {
                                if (i >= maxNumberOfRecordsToSave) {
                                    break
                                }

                                pushesToCache.add(result.result.items[i])
                                entries.add(
                                    CachingEntry(
                                        PushCacheLoader.PUSH_CACHE_KEY,
                                        result.result.items[i].id
                                    )
                                )
                            }

                            if (listOfIds.isNotEmpty()) {
                                //pulling initial pushes so remove all push for caching entry
                                PushDatabase.getDatabase(context).pushDataDAO().deleteByIds(listOfIds)
                                CommonsDatabase.getDatabase(context).cachingEntryDAO()
                                    .deleteAllByCachingKey(params.cachingKey)
                            }
                            PushDatabase.getDatabase(context).pushDataDAO().insert(pushesToCache)
                            CommonsDatabase.getDatabase(context).cachingEntryDAO().insert(entries)
                        }

                        if (result.result != null) {

                            val listOfPushObjects = arrayListOf<Push>()
                            result.result.items.forEach {
                                val pushModel = PushDataToPushConverter.convertPushDataToPush(it)
                                pushModel?.let {convertedPush ->
                                    listOfPushObjects.add(convertedPush)
                                }
                            }

                            //Report back to UI Thread the Push Items
                            CoroutineScope(Dispatchers.Main).launch {
                                callback?.onSuccess(listOfPushObjects, result.result.pagination)
                            }
                        }
                    }
                    else -> {

                        //Report back to UI Thread the Error as the request was not successful
                        CoroutineScope(Dispatchers.Main).launch {
                            callback?.onException(result.exception)
                        }
                    }
                }
            }catch (e : Exception){

                //Report back to UI Thread the Error as the request was not successful
                CoroutineScope(Dispatchers.Main).launch {
                    callback?.onException(FlybitsException(e))
                }
            }
        }
    }

}