package com.flybits.android.push

import android.content.Context
import android.os.Handler
import android.os.Looper
import com.flybits.android.push.api.FlyPushParsing
import com.flybits.android.push.db.converters.PushDataConverter
import com.flybits.android.push.db.converters.PushDataToPushConverter
import com.flybits.android.push.deserializations.DeserializePushToken
import com.flybits.android.push.exceptions.FlybitsPushException
import com.flybits.android.push.exceptions.InvalidPushLanguageCodeException
import com.flybits.android.push.models.Push
import com.flybits.android.push.models.PushToken
import com.flybits.android.push.models.newPush.DisplayablePush
import com.flybits.android.push.models.newPush.EventPush
import com.flybits.commons.library.SharedElementsFactory
import com.flybits.commons.library.api.FlyAway
import com.flybits.commons.library.api.results.BasicResult
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback
import com.flybits.commons.library.deserializations.IDeserializer
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.logging.Logger
import com.flybits.commons.library.models.internal.Result
import com.google.firebase.iid.FirebaseInstanceId
import org.json.JSONArray
import org.json.JSONObject
import java.util.*
import java.util.concurrent.Executors

/**
 * The `PushManager` is responsible for all Push related mechanisms such as registering for
 * push notifications, parsing push notifications, and communicating with the FCM servers.
 */
object PushManager {

    private const val API_PUSH_LANGUAGE = PushScope.ROOT + "/preferences/user"
    private const val API_PUSH_LANGUAGES = PushScope.ROOT + "/preferences/languages"

    /**
     * Disables Flybits-based push notifications including FCM and foreground notifications. This
     * process will remove the FCM token from the Flybits Push service as well as stopping the
     * foreground Push Service that is internal to the SDK.
     *
     * @param context The context of the activity that is trying to disable FCM push notifications.
     * @param callback The callback used to indicate whether or not disabling push was successful or
     * not. Default value is null.
     * @return The network request object that is triggers the `callback` based on the network
     * response.
     */
    @JvmStatic
    fun disablePush(context: Context, callback: BasicResultCallback? = null): BasicResult {
        return disablePush(context, SharedElementsFactory.get(context).getSavedJWTToken(), callback)
    }

    /**
     * Enable Flybits-based push notifications using FCM. This process will retrieve the FCM token
     * from Google and pass it to the Flybits Push Service.
     *
     * @param context The context of the activity that is trying to disable FCM push notifications.
     * @param properties The list of key-value properties that should be added to the Push Token.
     * Default value is null.
     * @param callback The callback used to indicate whether or not disabling push was successful or
     * not. Default value is null.
     */
    @JvmStatic
    fun enableFCMPush(
        context: Context,
        properties: HashMap<String, String>?,
        callback: BasicResultCallback? = null
    ) {
        FirebaseInstanceId.getInstance().instanceId.addOnSuccessListener { instanceIdResult ->
            val newToken = instanceIdResult.token
            enablePush(context, newToken, properties, callback)
        }
    }

    /**
     * Enable Flybits-based push notifications. This process will use the `token` and pass it
     * to the Flybits Push Service. This `token` can be an FCM token or any other one that is
     * necessary to register to push notifications.
     *
     * @param context The context of the activity that is trying to disable FCM push notifications.
     * @param token The push token that should be saved for the logged in user.
     * @param properties The list of key-value properties that should be added to the Push Token.
     * Default value is null
     * @param callback The callback used to indicate whether or not disabling push was successful or
     * not. Default value is null.
     * @param handler The handler that the callback methods will be invoked on.
     * Default value is a handler executing on the UI thread.
     * @return [BasicResult] that holds request information about the network request that was made.
     */
    @JvmStatic
    fun enablePush(
        context: Context,
        token: String,
        properties: HashMap<String, String>?,
        callback: BasicResultCallback? = null
        ,
        handler: Handler = Handler(Looper.getMainLooper())
    ): BasicResult {
        val executorService = Executors.newSingleThreadExecutor()
        val resultObject = BasicResult(callback, handler, executorService)
        executorService.execute {
            try {
                val resultSerializationObj = PushToken(token, properties)
                val deserialization = DeserializePushToken()
                val json = deserialization.toJson(resultSerializationObj)
                val pushResult: Result<Any> =
                    FlyAway.post(context, PushToken.API, json, null, "PushManager.enablePush", null)
                resultObject.setResult(pushResult)
            } catch (e: FlybitsException) {
                resultObject.setFailed(e)
            }
        }
        return resultObject
    }

    /**
     * Sets the language preference of the push notifications that are being delivered to the
     * device. The [languageCode] must in a 2-letter language code that follows the ISO 639-1
     * standard such as "en", "fr", "es", etc. If the language is not supported by your Flybits
     * project, it will default to "en".
     *
     * @param context The context of the application that trying to change language preference of
     * the user. Default is en
     * @param languageCode (optional) 2-letter ISO 639-1 language code.
     * @param callback The callback used to indicate whether or not setting the language was
     * successful or not. Default value is null.
     * @param handler The handler that the callback methods will be invoked on.
     * Default value is a handler executing on the UI thread.
     * @return [BasicResult] that holds request information about the network request that was made.
     */
    @JvmStatic
    fun setLanguage(
        context: Context, languageCode: String = "en", callback: BasicResultCallback? = null,
        handler: Handler = Handler(Looper.getMainLooper())
    ): BasicResult {

        val executorService = Executors.newSingleThreadExecutor()
        val resultObject = BasicResult(callback, handler, executorService)

        if (languageCode.length != 2) {
            resultObject.setFailed(InvalidPushLanguageCodeException())
        } else {
            return setLanguage(context, arrayListOf(languageCode), callback, handler)
        }
        return resultObject
    }

    /**
     * Sets a [List] of language preferences of the push notifications that are being delivered to the
     * device.
     * The [languageCodes] must follow the ISO 639 specifications.
     *
     * @param context The valid [Context] of the application that is trying to change language preferences of the user.
     * @param languageCodes The [List] of language codes compliant with ISO 639 specifications.
     * @param callback The [BasicResultCallback] used to indicate whether or not setting the language was successful
     * or not. Default value is null.
     * @param handler The [Handler] that the callback methods will be invoked on.
     * Default value is a handler executing on the UI thread.
     * @return [BasicResult] that holds request information about the network request that was made.
     */
    @JvmStatic
    fun setLanguage(
        context: Context, languageCodes: List<String>, callback: BasicResultCallback? = null,
        handler: Handler = Handler(Looper.getMainLooper())
    ): BasicResult {

        val executorService = Executors.newSingleThreadExecutor()
        val resultObject = BasicResult(callback, handler, executorService)

        executorService.execute {
            try {
                val jsonArray = JSONArray()
                for (code in languageCodes) {
                    jsonArray.put(code)
                }
                val json = JSONObject()
                json.put("languages", jsonArray)
                val pushResult: Result<Any> = FlyAway.put(
                    context, API_PUSH_LANGUAGES, json.toString(),
                    null, "PushManager.setLanguage", null
                )
                resultObject.setResult(pushResult)
            } catch (e: FlybitsException) {
                resultObject.setFailed(e)
            }
        }
        return resultObject
    }

    /**
     * Parse a FCM [Push] notification that is received from the Flybits platform. This push
     * notification is provided in a specific Flybits format which the SDK can parse. If the push
     * notification is not a Flybits notification and therefore cannot be parsed a
     * [FlybitsPushException] will be thrown within the
     * `PendingRequest.RequestCallbackWithResponse#onException(FlybitsException)`.
     *
     * @param mContext The [Context] of the application.
     * @param map The [Map] which contains the properties sent from the Flybits
     * platform. This should not be tempered with and should be sent directly to this
     * method from the FCM receiver.
     * @param callback The callback that initiated when the request is completed. It will contain
     * either a successful method or failure with a [FlybitsException] which
     * indicates the reason for failure. Default value is null.
     * @param <T> The `body` of the custom fields option. Some push notification may
     * provide additional fields with the request in order provide mechanisms for
     * deep-linking hence a unique structure can be provided.
     *
    </T> */
    @Deprecated(
        "Deprecated in version 2.3.2, will be removed in version 4.0.0",
        ReplaceWith(
            "parseDisplayablePushNotification() for Displayable Push and parseEventPushNotification" +
                    " for Event Push"
        )
    )
    @JvmStatic
    fun <T : IDeserializer<*>> parsePushNotification(
        mContext: Context,
        map: Map<*, *>,
        callback: ObjectResultCallback<Push>? = null
    ): Push? {
        try {
            val push = FlyPushParsing.parsePushNotification(mContext, map)
            callback?.onSuccess(push)
            return push
        } catch (e: FlybitsPushException) {
            callback?.onException(e)
        }
        return null
    }

    /**
     * Parse a FCM [DisplayablePush] notification that is received from the Flybits platform. This push
     * notification is provided in a specific Flybits format which the SDK can parse. If the DisplayablePush
     * notification is not a Flybits notification and therefore cannot be parsed a
     * [FlybitsPushException] will be thrown.
     *
     * @param map The [Map] which contains the properties sent from the Flybits
     * platform. This should not be tempered with and should be sent directly to this
     * method from the FCM receiver.
     * @param callback The callback that initiated when the request is completed. It will contain
     * either a successful method or failure with a [FlybitsPushException] which
     * indicates the reason for failure. Default value is null.
     **/
    @JvmStatic
    fun parseDisplayablePushNotification(
        map: Map<String, String>?,
        callback: ObjectResultCallback<DisplayablePush>
    ) {
        try {
            val pushData = PushDataConverter.convertMapToPushData(map as Map<String, String>)
            pushData?.let {
                val push = PushDataToPushConverter.convertPushDataToPush(it)
                push?.let { itPush ->
                    when (itPush) {
                        is DisplayablePush -> {
                            callback.onSuccess(itPush)
                        }
                        else -> callback.onException(
                            FlybitsPushException("Non Displayable Flybits Push.")
                        )
                    }
                } ?: callback.onException(
                    FlybitsPushException(
                        "Could Not Parse PushData into Push object. " +
                                "This should not happen. Please contact support@flybits.com for more details"
                    )
                )
            } ?: callback.onException(FlybitsPushException("Non-Flybits Push."))
        } catch (e: FlybitsPushException) {
            callback.onException(e)
        }
    }

    /**
     * Parse a FCM [EventPush] notification that is received from the Flybits platform. This push
     * notification is provided in a specific Flybits format which the SDK can parse. If the EventPush
     * notification is not a Flybits notification and therefore cannot be parsed a
     * [FlybitsPushException] will be thrown.
     *
     * @param map The [Map] which contains the properties sent from the Flybits
     * platform. This should not be tempered with and should be sent directly to this
     * method from the FCM receiver.
     * @param callback The callback that initiated when the request is completed. It will contain
     * either a successful method or failure with a [FlybitsPushException] which
     * indicates the reason for failure. Default value is null.
     **/
    @JvmStatic
    fun parseEventPushNotification(
        map: Map<String, String>?,
        callback: ObjectResultCallback<EventPush>
    ) {
        try {
            val pushData = PushDataConverter.convertMapToPushData(map as Map<String, String>)
            pushData?.let {
                val push = PushDataToPushConverter.convertPushDataToPush(it)
                push?.let { itPush ->
                    when (itPush) {
                        is EventPush -> {
                            callback.onSuccess(itPush)
                        }
                        else -> callback.onException(
                            FlybitsPushException("Non Event Flybits Push.")
                        )
                    }
                } ?: callback.onException(
                    FlybitsPushException(
                        "Could Not Parse PushData into Push object. " +
                                "This should not happen. Please contact support@flybits.com for more details"
                    )
                )
            } ?: callback.onException(FlybitsPushException("Non-Flybits Push Message."))
        } catch (e: FlybitsPushException) {
            callback.onException(e)
        }
    }

    /**
     * Disables Flybits-based push notifications including FCM and foreground notifications. This
     * process will remove the FCM token from the Flybits Push service as well as stopping the
     * foreground Push Service that is internal to the SDK.
     *
     * @param context The context of the activity that is trying to disable FCM push notifications.
     * @param jwttoken The jwt token associated with the user that the push notifications are being
     * disabled for.
     * @param callback The callback used to indicate whether or not disabling push was successful or
     * not. Default value is null.
     * @param handler The handler that the callback methods will be invoked on.
     * @return The network request object that is triggers the `callback` based on the network
     * response. Default value is a handler executing on the UI thread.
     */
    @JvmStatic
    fun disablePush(
        context: Context, jwttoken: String, callback: BasicResultCallback? = null
        , handler: Handler = Handler(Looper.getMainLooper())
    ): BasicResult {

        val executorService = Executors.newSingleThreadExecutor()
        val resultObject = BasicResult(callback, handler, executorService)
        executorService.execute {
            try {
                val mapOfHeaders = HashMap<String, String>()
                mapOfHeaders["X-Authorization"] = jwttoken
                val result = FlyAway.delete(
                    context,
                    PushToken.API,
                    mapOfHeaders,
                    "PushManager.disablePush",
                    null
                )
                resultObject.setResult(result)
            } catch (e: FlybitsException) {
                Logger.exception("PushManager.disablePush", e)
                resultObject.setFailed(e)
            }
        }
        return resultObject
    }
}
