package com.flybits.android.push.services

import android.app.PendingIntent
import android.content.Intent
import android.os.Handler
import android.os.Looper
import com.flybits.android.push.FlyNotification
import com.flybits.android.push.PushManager
import com.flybits.android.push.db.converters.PushDataConverter
import com.flybits.android.push.db.converters.PushDataToPushConverter
import com.flybits.android.push.models.newPush.DisplayablePush
import com.flybits.android.push.models.newPush.EventPush
import com.flybits.android.push.models.newPush.Push
import com.flybits.android.push.receiver.NotificationClickedReceiver
import com.flybits.commons.library.SharedElementsFactory
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.logging.Logger
import com.google.firebase.messaging.FirebaseMessagingService
import com.google.firebase.messaging.RemoteMessage
import java.util.*

const val EXTRA_PUSH_NOTIFICATION = "com.flybits.android.push.services.push_notification"

/**
 * Extend this class to automate the handling of Flybits push notifications. Make sure to register the class
 * in the AndroidManifest <application> block, like so:
 *
 * <service
 *      android:name="name.of.the.class.extending.PushService">
 *      <intent-filter>
 *          <action android:name="com.google.firebase.MESSAGING_EVENT" />
 *      </intent-filter>
 * </service>
 *
 */
abstract class PushService : FirebaseMessagingService() {


    private val ErrorConvertingPushDataToPush = "Could Not Parse PushData into Push object. " +
            "This should not happen. Please contact support@flybits.com for more details"

    /**
     * This function will be invoked whenever a non Flybits push notification is received.
     *
     * @param remoteMessage The [RemoteMessage] associated with the non Flybits push notification.
     */
    abstract fun onNonFlybitsPushReceived(remoteMessage: RemoteMessage)

    /**
     *
     * @param push [Push] for which the icon will be displayed.
     *
     * @return id of the icon resource that will be displayed with notifications.
     */
    abstract fun getNotificationIconRes(push: DisplayablePush): Int

    /**
     *
     * Please Note : The Notification Channel creation is NOT required for devices with API less than 26,
     * notifications would be displayed even if non existing channel id is returned.
     * and
     * The Notification Channel creation is required for devices with API more than or equal to 26,
     * notifications would NOT be displayed if non existing channel id is returned.
     *
     * @param push The [Push] provided to extract information in order to construct the notification.
     *
     * @return id of already existing notification channel.
     */
    abstract fun getNotificationChannelId(push: Push): String

    override fun onMessageReceived(remoteMessage: RemoteMessage?) {
        super.onMessageReceived(remoteMessage)

        Logger.d("Push received: ${remoteMessage?.data}")

        if (remoteMessage != null) {
            try {

                val pushData = PushDataConverter.convertMapToPushData(remoteMessage.data)
                pushData?.let {

                    //We should consider saving the push here into the DB
                    val push = PushDataToPushConverter.convertPushDataToPush(it)
                    push?.let { itPush ->
                        when (itPush) {
                            is DisplayablePush -> {
                                setChannelIdPref(itPush)
                                onDisplayableFlybitsPushReceived(itPush)
                            }
                            is EventPush -> onEntityFlybitsPushReceived(itPush)
                        }
                    } ?: Logger.e(ErrorConvertingPushDataToPush)
                } ?: onNonFlybitsPushReceived(remoteMessage)
            } catch (e: FlybitsException) {
                Logger.exception("ConciergeMessagingService.onMessageReceived()", e)
            }
        }
    }

    private fun setChannelIdPref(displayablePush: DisplayablePush) {
        val sharedPreferences =
            SharedElementsFactory.get(applicationContext)
        val pushChannelId = getNotificationChannelId(displayablePush)
        if (pushChannelId.isNotBlank())
            sharedPreferences.setNotificationChannel(pushChannelId)
    }

    override fun onNewToken(token: String?) {
        super.onNewToken(token)
        if (token != null && SharedElementsFactory.get(applicationContext).getSavedJWTToken()
                .isNotEmpty()
        ) {
            PushManager.enablePush(
                applicationContext,
                token,
                HashMap(),
                null,
                Handler(Looper.getMainLooper())
            )
        }
    }

    /**
     * This function indicates what intent should be fired off when the notification is clicked.
     * If you override the [onDisplayableFlybitsPushReceived] method you may want to implement the
     * notification click yourself through the setPendingIntent method. However, if you do this
     * yourself you will not be recording notification click on Flybits. If you want to record the
     * notification click on Flybits you will result of the [PendingIntent] to implement the
     * following yourself.
     *
     * val pushAnalytics = PushAnalytics(applicationContext)
     * pushAnalytics.trackEngaged(it as DisplayablePush)
     *
     * @param push The [DisplayablePush] that is received from the Flybits server.
     * @return The [PendingIntent] that is responsible for launching the application and records the
     * notification click in the Flybits platform.
     */
    open fun onNotificationClick(push: DisplayablePush): PendingIntent {

        val intent = Intent(applicationContext, NotificationClickedReceiver::class.java).apply {
            putExtra(EXTRA_PUSH_NOTIFICATION, push)
        }
        return PendingIntent.getBroadcast(
            applicationContext
            , push.id.hashCode(), intent, PendingIntent.FLAG_UPDATE_CURRENT
        )
    }

    /**
     * This function will be invoked whenever a notification is received from the Flybits server
     * with the intention of being displayed to the end user.
     *
     * @param push [Push] received from Flybits
     */
    open fun onDisplayableFlybitsPushReceived(push: DisplayablePush) {
        FlyNotification.Builder(
            applicationContext,
            push,
            getNotificationChannelId(push),
            getNotificationIconRes(push)
        )
            .setPendingIntent(onNotificationClick(push))
            .build()
            .show()
    }

    /**
     * This function will be invoked whenever a Flybits push notification is received
     * indicating that a `PushEntity` has been either created (`PushAction.ADDED`) or
     * updated `PushAction.UPDATED` or deleted `PushAction.REMOVED`.
     *
     * @param push The [Push] which contains either the Create, Update or Delete `PushAction`.
     *
     */
    open fun onEntityFlybitsPushReceived(push: EventPush) {}
}