package com.flybits.android.push.models.newPush

import android.os.Parcel
import android.os.Parcelable
import org.json.JSONObject

/**
 * This class is used to represent a the root push object obtained from the Flybits Push Server through
 * FCM after conversion
 */

sealed class Push : Parcelable {

    val timestamp: Long
    val id: String

    /**
     * Constructor used to initiate the {@code Push} object.
     *
     * @param id The unique identifier for this {@code Push} object.
     * @param timestamp The timestamp, in milliseconds, representing when the push notification was
     *                  created.
     */
    constructor(timestamp: Long, id: String) {
        this.timestamp = timestamp
        this.id = id
    }

    /**
     * Constructor used for un-flattening a {@code Push} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code Push} parcel.
     */
    constructor(parcel: Parcel) {
        timestamp = parcel.readLong()
        id = parcel.readString() ?: ""
    }

    override fun writeToParcel(parcel: Parcel, flags: Int) {
        parcel.writeLong(timestamp)
        parcel.writeString(id)
    }

    override fun describeContents(): Int {
        return 0
    }

    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (javaClass != other?.javaClass) return false

        other as Push

        if (id != other.id) return false

        return true
    }

    override fun hashCode(): Int {
        return id.hashCode()
    }
}


/**
 * This class is used to represent a the event push object obtained from the Flybits Push Server through
 * FCM after conversion
 */

class EventPush : Push {

    val event: Event

    /**
     * Constructor used to initiate the {@code Push} object.
     *
     * @param id The unique identifier for this {@code Push} object.
     * @param timestamp The timestamp, in milliseconds, representing when the push notification was
     *                  created.
     * @param event The type of event : RELEVANT_CONTENT_CHANGE |.
     */
    constructor(
        timestamp: Long,
        id: String,
        event: Event
    ) : super(timestamp, id) {
        this.event = event
    }

    /**
     * Constructor used for un-flattening a {@code Push} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code Push} parcel.
     */
    constructor(parcel: Parcel) : super(parcel) {
        event = Event.valueOf(parcel.readString() ?: "")
    }

    override fun writeToParcel(parcel: Parcel, flags: Int) {
        super.writeToParcel(parcel, flags)
        parcel.writeString(this.event.name)
    }

    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (javaClass != other?.javaClass) return false

        other as EventPush

        if (id != other.id) return false

        return true
    }

    override fun toString(): String {
        return "EventPush{id : $id, event : ${event.name}, timestamp : $timestamp}"
    }

    companion object CREATOR : Parcelable.Creator<EventPush> {
        override fun createFromParcel(parcel: Parcel): EventPush {
            return EventPush(parcel)
        }

        override fun newArray(size: Int): Array<EventPush?> {
            return arrayOfNulls(size)
        }
    }

}

/**
 * enum class for the types of events possible in Event Push Type
 */
enum class Event {
    RELEVANT_CONTENT_CHANGE
}

/**
 * This class is used to represent a the root displayable push object obtained from the Flybits Push Server through
 * FCM after conversion
 */

sealed class DisplayablePush : Push {
    val title: String?
    val message: String?
    val parentId: String


    /**
     * Constructor used to initiate the {@code Push} object.
     *
     * @param id The unique identifier for this {@code Push} object.
     * @param timestamp The timestamp, in milliseconds, representing when the push notification was
     *                  created.
     * @param parentId the root id of the template notification that we created this {@code Push} notification from
     * @param title The title of the displayable push notification.
     * @param message The message of the displayable push notification.
     */
    constructor(
        timestamp: Long,
        id: String,
        parentId: String,
        title: String?,
        message: String?
    ) : super(timestamp, id) {
        this.title = title
        this.message = message
        this.parentId = parentId
    }

    /**
     * Constructor used for un-flattening a {@code Push} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code Push} parcel.
     */
    constructor(parcel: Parcel) : super(parcel) {
        title = parcel.readString() ?: ""
        message = parcel.readString() ?: ""
        parentId = parcel.readString() ?: ""
    }

    override fun writeToParcel(parcel: Parcel, flags: Int) {
        super.writeToParcel(parcel, flags)
        parcel.writeString(title)
        parcel.writeString(message)
        parcel.writeString(parentId)
    }

}

/**
 * This class is used to represent a the basic displayable push object obtained from the Flybits Push Server through
 * FCM after conversion
 */

class BasicPush : DisplayablePush {

    /**
     * Constructor used to initiate the {@code Push} object.
     *
     * @param id The unique identifier for this {@code Push} object.
     * @param timestamp The timestamp, in milliseconds, representing when the push notification was
     *                  created.
     * @param parentId the root id of the template notification that we created this {@code Push} notification from
     * @param title The title of the displayable push notification.
     * @param message The message of the displayable push notification.
     */

    constructor(
        timestamp: Long,
        id: String,
        parentId: String,
        title: String?,
        message: String?
    ) : super(timestamp, id, parentId, title, message)

    /**
     * Constructor used for un-flattening a {@code Push} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code Push} parcel.
     */

    constructor(parcel: Parcel) : super(parcel)

    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (javaClass != other?.javaClass) return false

        other as BasicPush

        if (id != other.id) return false

        return true
    }

    override fun toString(): String {
        return "BasicPush{id : $id, timestamp : $timestamp, pushRequestId: $parentId, title : $title, message : $message}"
    }

    companion object CREATOR : Parcelable.Creator<BasicPush> {
        override fun createFromParcel(parcel: Parcel): BasicPush {
            return BasicPush(parcel)
        }

        override fun newArray(size: Int): Array<BasicPush?> {
            return arrayOfNulls(size)
        }
    }

}

/**
 * This class is used to represent a the content displayable push object obtained from the Flybits Push Server through
 * FCM after conversion
 */

class ContentPush : DisplayablePush {

    val contentId: String

    /**
     * Constructor used to initiate the {@code Push} object.
     *
     * @param id The unique identifier for this {@code Push} object.
     * @param timestamp The timestamp, in milliseconds, representing when the push notification was
     *                  created.
     * @param parentId the root id of the template notification that we created this {@code Push} notification from
     * @param title The title of the displayable push notification.
     * @param message The message of the displayable push notification.
     * @param contentId The id of the content in the displayable push notification.
     */

    constructor(
        timestamp: Long,
        id: String,
        parentId: String,
        title: String?,
        message: String?,
        contentId: String
    ) : super(timestamp, id, parentId, title, message) {
        this.contentId = contentId
    }

    /**
     * Constructor used for un-flattening a {@code Push} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code Push} parcel.
     */

    constructor(parcel: Parcel) : super(parcel) {
        contentId = parcel.readString() ?: ""
    }

    override fun writeToParcel(parcel: Parcel, flags: Int) {
        super.writeToParcel(parcel, flags)
        parcel.writeString(contentId)
    }

    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (javaClass != other?.javaClass) return false

        other as ContentPush

        if (id != other.id) return false

        return true
    }

    override fun toString(): String {
        return "ContentPush{id : $id, contentId : $contentId, timestamp : $timestamp, pushRequestId: $parentId, title : $title, message : $message}"
    }

    companion object CREATOR : Parcelable.Creator<ContentPush> {
        override fun createFromParcel(parcel: Parcel): ContentPush {
            return ContentPush(parcel)
        }

        override fun newArray(size: Int): Array<ContentPush?> {
            return arrayOfNulls(size)
        }
    }

}

/**
 * This class is used to represent a the web link displayable push object obtained from the Flybits Push Server through
 * FCM after conversion
 */

class WeblinkPush : DisplayablePush {
    val url: String

    /**
     * Constructor used to initiate the {@code Push} object.
     *
     * @param id The unique identifier for this {@code Push} object.
     * @param timestamp The timestamp, in milliseconds, representing when the push notification was
     *                  created.
     * @param parentId the root id of the template notification that we created this {@code Push} notification from
     * @param title The title of the displayable push notification.
     * @param message The message of the displayable push notification.
     * @param url The url of the web link in the displayable push notification.
     */


    constructor(
        timestamp: Long,
        id: String,
        parentId: String,
        title: String?,
        message: String?,
        url: String
    ) : super(timestamp, id, parentId, title, message) {
        this.url = url
    }

    /**
     * Constructor used for un-flattening a {@code Push} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code Push} parcel.
     */

    constructor(parcel: Parcel) : super(parcel) {
        url = parcel.readString() ?: ""
    }

    override fun writeToParcel(parcel: Parcel, flags: Int) {
        super.writeToParcel(parcel, flags)
        parcel.writeString(url)
    }

    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (javaClass != other?.javaClass) return false

        other as WeblinkPush

        if (id != other.id) return false

        return true
    }

    override fun toString(): String {
        return "WeblinkPush{id : $id, url : $url, timestamp : $timestamp, pushRequestId: $parentId, title : $title, message : $message}"
    }

    companion object CREATOR : Parcelable.Creator<WeblinkPush> {
        override fun createFromParcel(parcel: Parcel): WeblinkPush {
            return WeblinkPush(parcel)
        }

        override fun newArray(size: Int): Array<WeblinkPush?> {
            return arrayOfNulls(size)
        }
    }

}

/**
 * This class is used to represent a the custom displayable push object obtained from the Flybits Push Server through
 * FCM after conversion
 */

class CustomDisplayablePush : DisplayablePush {

    val properties: JSONObject
    val notificationTypeId: String

    /**
     * Constructor used to initiate the {@code Push} object.
     *
     * @param id The unique identifier for this {@code Push} object.
     * @param timestamp The timestamp, in milliseconds, representing when the push notification was
     *                  created.
     * @param parentId the root id of the template notification that we created this {@code Push} notification from
     * @param notificationTypeId the identifier of the custom body of the {@code Push} notification
     * @param title The title of the displayable push notification.
     * @param message The message of the displayable push notification.
     * @param properties The object in the custom displayable push notification.
     */

    constructor(
        timestamp: Long,
        id: String,
        parentId: String,
        notificationTypeId: String,
        title: String?,
        message: String?,
        properties: JSONObject
    ) : super(timestamp, id, parentId, title, message) {
        this.properties = properties
        this.notificationTypeId = notificationTypeId
    }

    /**
     * Constructor used for un-flattening a {@code Push} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code Push} parcel.
     */

    constructor(parcel: Parcel) : super(parcel) {
        properties = JSONObject(parcel.readString())
        notificationTypeId = parcel.readString() ?: ""
    }

    override fun writeToParcel(parcel: Parcel, flags: Int) {
        super.writeToParcel(parcel, flags)
        parcel.writeString(properties.toString())
        parcel.writeString(notificationTypeId)
    }

    override fun equals(other: Any?): Boolean {
        if (this === other) return true
        if (javaClass != other?.javaClass) return false

        other as CustomDisplayablePush

        if (id != other.id) return false

        return true
    }

    override fun toString(): String {
        return "CustomDisplayablePush{id : $id, properties : $properties, timestamp : $timestamp, pushRequestId: $parentId, pushPayloadTypeId : $notificationTypeId, title : $title, message : $message}"
    }

    companion object CREATOR : Parcelable.Creator<CustomDisplayablePush> {
        override fun createFromParcel(parcel: Parcel): CustomDisplayablePush {
            return CustomDisplayablePush(parcel)
        }

        override fun newArray(size: Int): Array<CustomDisplayablePush?> {
            return arrayOfNulls(size)
        }
    }

}