package com.flybits.android.push.db.converters

import android.arch.persistence.room.TypeConverter
import com.flybits.android.push.models.PushData
import org.json.JSONException
import org.json.JSONObject
import java.math.BigDecimal

object PushDataConverter{

    private const val MAP_ID = "id"
    private const val MAP_TIMESTAMP = "timestamp"
    private const val MAP_VERSION = "version"
    private const val MAP_BODY = "body"
    private const val MAP_REQUEST_ID = "pushRequestId"
    private const val MAP_PUSH_PAYLOAD_ID = "pushPayloadTypeId"
    private const val MAP_ALERT = "alert"
    private const val MAP_TITLE = "title"

    @TypeConverter
    @JvmStatic
    fun convertMapToPushData(map : Map<String, String>) : PushData? {

        if (!map.containsKey(MAP_ID) || !map.containsKey(MAP_TIMESTAMP) || !map.containsKey(
                MAP_VERSION)){
            return null
        }


        val version = try{
            Integer.parseInt(map[MAP_VERSION]?:"")
        }catch (e : NumberFormatException) {
            2
        }

        val timestamp = BigDecimal(map[MAP_TIMESTAMP]).toLong()

        val body = map[MAP_BODY]
        val jsonBody = try{
            JSONObject(body)
        }catch (e : JSONException){
            JSONObject()
        }

        val pushRequestId = jsonBody.optString(MAP_REQUEST_ID)
        val pushPayloadTypeId =
            if (jsonBody.has(MAP_PUSH_PAYLOAD_ID))
                jsonBody.getString(MAP_PUSH_PAYLOAD_ID)
            else
                null

        val title =
            if (map.containsKey(MAP_TITLE))
                map[MAP_TITLE]
            else
                null

        val alert =
            if (map.containsKey(MAP_ALERT))
                map[MAP_ALERT]
            else
                null

        return PushData(
            map.getValue(MAP_ID),
            pushRequestId,
            timestamp,
            version,
            pushPayloadTypeId,
            jsonBody,
            title,
            alert
        )
    }

    @TypeConverter
    @JvmStatic
    fun convertJsonToPushData(json : String) : PushData? {

        val jsonObject = try {
            JSONObject(json)
        } catch (e : JSONException){
            return null
        }

        if (!jsonObject.has(MAP_ID) || !jsonObject.has(MAP_TIMESTAMP) || !jsonObject.has(MAP_VERSION)){
            return null
        }

        val version = jsonObject.optInt(MAP_VERSION, 2)
        val timestamp = jsonObject.getLong(MAP_TIMESTAMP)
        val title = jsonObject.optString(MAP_TITLE, null)
        val alert = jsonObject.optString(MAP_ALERT, null)

        val body =
            if (jsonObject.has(MAP_BODY)) {
                jsonObject.optJSONObject(MAP_BODY)
            }else {
                JSONObject("{}")
            }

        val pushRequestId = body.optString(MAP_REQUEST_ID, null)
        val pushPayloadTypeId = body.optString(MAP_PUSH_PAYLOAD_ID, null)

        return PushData(
            jsonObject.getString(MAP_ID),
            pushRequestId,
            timestamp,
            version,
            pushPayloadTypeId,
            body,
            title,
            alert
        )
    }
}