package com.flybits.android.push.models

import android.content.Context
import android.os.Handler
import android.os.Looper
import android.os.Parcel
import android.os.Parcelable

import com.flybits.android.push.PushScope
import com.flybits.android.push.deserializations.DeserializePushToken
import com.flybits.commons.library.api.FlyAway
import com.flybits.commons.library.api.results.ObjectResult
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.models.internal.Result

import java.util.concurrent.Executors

/**
 * This class is used to define a FCM push token which should be transferred from the device, once
 * it is obtained from Google, to the Flybits Push Server. Through this `PushToken` class the
 * Flybits Push Server will be able to know to which `User` to send Push notification to
 * based on specific Server and Business logic.
 *
 * @param deviceToken The FCM token associated to this device.
 * @param type The integer value that indicates which type of device this token represents, 1 is for Google.
 * @param data The [HashMap] containing additional information associated to the [PushToken].
 * @param pushVersion The version of the push payload that this version of the Flybits Push SDK supports. Is currently 2.
 */
class PushToken(val deviceToken: String
                , val data: Map<String, String> = emptyMap()
                , val type: Long = 1
                , val pushVersion: Int = 2) : Parcelable {

    /**
     * Constructor used for un-flattening a [Push] [Parcel].
     *
     * @param parcel The [Parcel] that contains the un-flattened [Push] [Parcel].
     */
    constructor(parcel: Parcel) : this(parcel.readString() ?: ""
            , parcel.readHashMap(String::class.java.classLoader) as Map<String, String>? ?: emptyMap()
            , parcel.readLong()
            , parcel.readInt())

    /**
     * Constructor for instantiating [PushToken] with only the deviceToken. Added for backwards compatibility
     * when used in Java.
     *
     * @param deviceToken The FCM token associated to this device.
     * @param data The [Map] containing additional information associated to the [PushToken].
     */
    constructor(deviceToken: String): this(deviceToken, emptyMap())

    /**
     * Constructor for instantiating [PushToken] with only the deviceToken, and the data map.
     * Added for backwards compatibility when used in Java.
     *
     * @param deviceToken The FCM token associated to this device.
     * @param data
     */
    constructor(deviceToken: String, data: Map<String, String>?): this(deviceToken, data ?: emptyMap<String,String>())

    /**
     * Describe the kinds of special objects contained in this Parcelable's marshalled representation.
     *
     * @return a bitmask indicating the set of special object types marshalled by the Parcelable.
     */
    override fun describeContents(): Int {
        return 0
    }


    /**
     * Flatten this [PushToken] into a [Parcel].
     *
     * @param out The Parcel in which the [PushToken] object should be written.
     * @param flags Additional flags about how the DateOfBirth object should be written.
     * May be 0 or [.PARCELABLE_WRITE_RETURN_VALUE].
     */
    override fun writeToParcel(out: Parcel, flags: Int) {
        out.writeString(deviceToken)
        out.writeMap(data)
        out.writeLong(type)
        out.writeInt(pushVersion)
    }

    companion object {

        const val API = PushScope.ROOT + "/token"

        /**
         * [Parcelable.Creator] that instantiates [PushToken] objects.
         */
        @JvmField
        val CREATOR: Parcelable.Creator<PushToken> = object : Parcelable.Creator<PushToken> {
            override fun createFromParcel(parcel: Parcel): PushToken {
                return PushToken(parcel)
            }

            override fun newArray(size: Int): Array<PushToken?> {
                return arrayOfNulls(size)
            }
        }

        /**
         * Get the [PushToken] that is associated to the logged in `User`.
         *
         * @param context The [Context] of the application that is retrieving the FCM Push Token.
         * @param callback The [ObjectResultCallback] that indicates whether or not the GET request was successful or
         * it failed.
         * @return The [ObjectResult] that returns the [PushToken] within the [ObjectResultCallback.onSuccess]
         * method.
         */
        @JvmStatic
        fun get(context: Context, callback: ObjectResultCallback<PushToken>): ObjectResult<PushToken> {

            val handler = Handler(Looper.getMainLooper())
            val executorService = Executors.newSingleThreadExecutor()
            val query = ObjectResult(callback, handler, executorService)

            executorService.execute {
                try {
                    val result =
                    FlyAway.get(context, API, DeserializePushToken(), "PushToken.get", PushToken::class.java)
                    query.setResult(result as Result<PushToken>)
                } catch (e: FlybitsException) {
                    query.setFailed(e)
                }
            }
            return query
        }
    }
}
