package com.flybits.android.push.api;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;

import com.flybits.android.push.deserializations.DeserializePushPreferences;
import com.flybits.android.push.models.Push;
import com.flybits.commons.library.api.FlyAway;
import com.flybits.commons.library.api.results.BasicResult;
import com.flybits.commons.library.api.results.ObjectResult;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.models.internal.Result;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import static com.flybits.android.push.PushScope.ROOT;

/**
 * The {@code PushPreferenceManager} is responsible actions that can be associated to the
 * retrieval of {@link Push} preferences that indicate any additional information associated to the
 * {@code PushToken} that is sent to the Server for processing whenever the {@code User} logs in.
 */
public class PushPreferenceManager {

    static final     String API                     = ROOT + "/preferences";

    /**
     * Subscribing to a {@code topic} allows a user to receive push notifications about a specific
     * {@code topic}. If you pass the keyword "restaurants" as the {@code topic}, you will receive
     * push notification about the term topic.
     *
     * @param context The context of the application.
     * @param topic The topic that should be subscribed to.
     * @param callback The callback that initiated when the request is completed. It will contain
     *                 either a successful method or failure with a {@code FlybitsException} which
     *                 indicates the reason for failure.
     * @return The {@link ExecutorService} which can be shutdown by the application developer in
     * case the request is taking too long or the user no longer needs the result from the request.
     */
    public static BasicResult subscribe(@NonNull final Context context, @NonNull final String topic, @NonNull final BasicResultCallback callback){
        final Handler handler   = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult query = new BasicResult(callback, handler, executorService);
        executorService.execute(new Runnable() {
            public void run() {
                try {
                    //Step 1: Get All Topics that you are unsubscribed from
                    final Result resultGetSubscriptions  = FlyAway.get(context, API, new DeserializePushPreferences(), "PushPreferenceManager.subscribe1", String[].class);
                    if (resultGetSubscriptions.getStatus() == RequestStatus.COMPLETED){
                        //Step 2: Remove subscription (REMEMBER by removing we are actually Adding --> Yikes I know it doesn't sound right ... but it is ...)
                        List<String> listOfUnsubscribedTopic = new ArrayList<>(Collections.singletonList((String) resultGetSubscriptions.getResult()));
                        if (listOfUnsubscribedTopic.contains(topic)) {
                            listOfUnsubscribedTopic.remove(topic);
                            String[] listOfTopics = listOfUnsubscribedTopic.toArray(new String[listOfUnsubscribedTopic.size()]);
                            final String json = new DeserializePushPreferences().toJson(listOfTopics);
                            final Result resultSubscribe  = FlyAway.post(context, API, json, null, "PushPreferenceManager.subscribe2", null);
                            query.setResult(resultSubscribe);
                        }else{
                            query.setResult(resultGetSubscriptions);
                        }
                    }else{
                        query.setResult(resultGetSubscriptions);
                    }
                } catch (final FlybitsException e) {
                    query.setFailed(e);
                }
            }
        });
        return query;
    }

    /**
     * Unsubscribing from a {@code topic} blocks all push notifications about a specific
     * {@code topic} so that a user would not receive such notifications . If you pass the keyword
     * "restaurants" as the {@code topic}, the device will NOT receive any push notification
     * associated to the {@code topic}.
     *
     * @param context The context of the application.
     * @param topic The topic that should be unsubscribed from.
     * @param callback The callback that initiated when the request is completed. It will contain
     *                 either a successful method or failure with a {@code FlybitsException} which
     *                 indicates the reason for failure.
     * @return The {@link ExecutorService} which can be shutdown by the application developer in
     * case the request is taking too long or the user no longer needs the result from the request.
     */
    public static BasicResult unsubscribe(@NonNull final Context context, @NonNull final String topic, @NonNull final BasicResultCallback callback){
        final Handler handler   = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult query = new BasicResult(callback, handler, executorService);
        executorService.execute(new Runnable() {
            public void run() {
                try {
                    //Step 1: Get All Topics that you are unsubscribed from
                    final Result resultGetSubscriptions  = FlyAway.get(context, API, new DeserializePushPreferences(), "PushPreferenceManager.unsubscribe1", String[].class);
                    if (resultGetSubscriptions.getStatus() == RequestStatus.COMPLETED){
                        //Step 2: Add an unsubscribed item (REMEMBER by removing we are actually Adding --> Yikes I know it doesn't sound right ... but it is ...
                        ArrayList<String> listOfUnsubscribedTopic = new ArrayList<String>(Collections.<String>singletonList((String) resultGetSubscriptions.getResult()));
                        if (!listOfUnsubscribedTopic.contains(topic)) {
                            listOfUnsubscribedTopic.add(topic);
                            String[] listOfTopics = listOfUnsubscribedTopic.toArray(new String[listOfUnsubscribedTopic.size()]);
                            final String json = new DeserializePushPreferences().toJson(listOfTopics);
                            final Result resultUnsubscribe  = FlyAway.post(context, API, json, null, "PushPreferenceManager.unsubscribe2", null);
                            query.setResult(resultUnsubscribe);
                        }else{
                            query.setResult(resultGetSubscriptions);
                        }
                    }else{
                        query.setResult(resultGetSubscriptions);
                    }
                } catch (final FlybitsException e) {
                    query.setFailed(e);
                }
            }
        });
        return query;
    }

    /**
     * Clear the {@code PushPreferences} list which means that the connected user will receive all
     * push notification and no push notifications will be filtered out.
     *
     * @param context The state of the application.
     * @param callback The callback that initiated when the request is completed. It will contain
     *                 either a successful method or failure with a
     *                 {@code FlybitsException} which indicates the reason for failure.
     * @return The {@link ExecutorService} which can be shutdown by the application developer in
     * case the request is taking too long or the user no longer needs the result from the request.
     */
    public static BasicResult clear(@NonNull final Context context,
                                    @NonNull final BasicResultCallback callback){

        final Handler handler   = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult query = new BasicResult(callback, handler, executorService);
        executorService.execute(new Runnable() {
            public void run() {

                try {
                    final Result result = FlyAway.delete(context, API, "PushPreferenceManager.clear", null);
                    query.setResult(result);
                }catch(final FlybitsException e){
                    query.setFailed(e);
                }
            }
        });
        return query;
    }

    /**
     * Retrieve all the {@code PushPreferences} that are stored for the connected user. It is
     * important to make sure that the user is in fact connected to Flybits. This can be achieved
     * through the {@code FlybitsManager#connect()} function.
     *
     * @param context The state of the application.
     * @param callback The callback that initiated when the request is completed. It will contain
     *                 either a successful method or failure with a
     *                 {@code FlybitsException} which indicates the reason for failure.
     * @return The {@link ExecutorService} which can be shutdown by the application developer in
     * case the request is taking too long or the user no longer needs the result from the request.
     */
    public static ObjectResult<String[]> get(@NonNull final Context context,
                                             @NonNull final ObjectResultCallback<String[]> callback) {

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ObjectResult<String[]> query    = new ObjectResult<String[]>(callback, handler, executorService);
        executorService.execute(new Runnable() {
            public void run() {

                try {
                    final Result<String[]> result  = FlyAway.get(context, API, new DeserializePushPreferences(), "PushPreferenceManager.get", String[].class);
                    query.setResult(result);
                } catch (final FlybitsException e) {
                    query.setFailed(e);
                }
            }
        });
        return query;
    }

    /**
     * The function will override the current list of unsubscribed items used to filter push
     * notifications for. For example, if the user does not want to receive notification about
     * "restaurants", then "restaurants" should be including in the {@code listOfOptions} parameter.
     * If this is done the connected user will not receive any push notification that have been
     * labeled with the "restaurants" label.
     *
     * @param context The context of the application.
     * @param keywords The list of options that should be used to filter push notification for.
     * @param callback The callback that initiated when the request is completed. It will contain
     *                 either a successful method or failure with a
     *                 {@code FlybitsException} which indicates the reason for failure.
     * @return The {@link ExecutorService} which can be shutdown by the application developer in
     * case the request is taking too long or the user no longer needs the result from the request.
     */
    public static BasicResult override(@NonNull final Context context, @NonNull final ArrayList<String> keywords,
                                   final BasicResultCallback callback) {

        final Handler handler   = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult query = new BasicResult(callback, handler, executorService);
        executorService.execute(new Runnable() {
            public void run() {

                try {
                    String[] listOfTopics = keywords.toArray(new String[keywords.size()]);
                    final String json = new DeserializePushPreferences().toJson(listOfTopics);
                    final Result result = FlyAway.post(context, API, json, null, "PushPreferenceManager.post", null);
                    query.setResult(result);
                } catch (final FlybitsException e) {
                    query.setFailed(e);
                }
            }
        });
        return query;
    }
}
