package com.flybits.android.push.db.converters

import android.arch.persistence.room.TypeConverter
import com.flybits.android.push.BuildConfig
import com.flybits.android.push.models.PushData
import com.flybits.android.push.models.newPush.*
import com.flybits.commons.library.logging.Logger

internal object PushDataToPushConverter{

    private const val ATTRIBUTE_URL = "url"
    private const val ATTRIBUTE_CONTENT_ID = "contentId"
    private const val ATTRIBUTE_CONTENT_VALUE = "value"

    const val ENTITY_RELEVANT_CONTENT = "entityEvent"

    const val ERROR_NULL_REQUEST_ID = "You received a DisplablePush from Flybits that does not contain a RequestID. This should not happen. Please contact support@flybits.com for more details"
    const val ERROR_INVALID_WEB = "You received a WeblinkPush from Flybits that does not contain a url. This should not happen. Please contact support@flybits.com for more details"
    const val ERROR_INVALID_CONTENT = "You received a ContentPush from Flybits that does not contain a Content ID. This should not happen. Please contact support@flybits.com for more details"

    @TypeConverter
    @JvmStatic
    fun convertPushDataToPush(pushData : PushData) : Push? {
        when (pushData.pushPayloadTypeId){
            null -> {
                if (pushData.pushRequestId == null){
                    Logger
                        .e(ERROR_NULL_REQUEST_ID)
                    return null
                }

                return BasicPush(
                    pushData.timestamp,
                    pushData.id,
                    pushData.pushRequestId,
                    pushData.title,
                    pushData.message
                )
            }
            BuildConfig.METADATA_ID_WEB -> {
                val jsonBody = pushData.body
                if (!jsonBody.has(ATTRIBUTE_URL)){
                    Logger
                        .e(ERROR_INVALID_WEB)
                    return null
                }
                if (pushData.pushRequestId == null){
                    Logger
                        .e(ERROR_NULL_REQUEST_ID)
                    return null
                }
                return WeblinkPush(
                    pushData.timestamp,
                    pushData.id,
                    pushData.pushRequestId,
                    pushData.title,
                    pushData.message,
                    jsonBody.getString(ATTRIBUTE_URL)
                )
            }
            BuildConfig.METADATA_ID_CONTENT ->{
                val jsonBody = pushData.body
                if (!jsonBody.has(ATTRIBUTE_CONTENT_ID)){
                    Logger
                        .e(ERROR_INVALID_CONTENT)
                    return null
                }
                if (pushData.pushRequestId == null){
                    Logger
                        .e(ERROR_NULL_REQUEST_ID)
                    return null
                }

                val jsonContent = jsonBody.getJSONObject(ATTRIBUTE_CONTENT_ID)
                if (!jsonContent.has(ATTRIBUTE_CONTENT_VALUE)){
                    Logger
                        .e(ERROR_INVALID_CONTENT)
                    return null
                }

                return ContentPush(
                    pushData.timestamp,
                    pushData.id,
                    pushData.pushRequestId,
                    pushData.title,
                    pushData.message,
                    jsonContent.getString(ATTRIBUTE_CONTENT_VALUE)
                )

            }
            ENTITY_RELEVANT_CONTENT -> {
                return EventPush(
                    pushData.timestamp,
                    pushData.id,
                    Event.RELEVANT_CONTENT_CHANGE)
            }
            else->{
                if (pushData.pushRequestId == null){
                    Logger
                        .e(ERROR_NULL_REQUEST_ID)
                    return null
                }

                return CustomDisplayablePush(
                    pushData.timestamp,
                    pushData.id,
                    pushData.pushRequestId,
                    pushData.pushPayloadTypeId,
                    pushData.title,
                    pushData.message,
                    pushData.body
                )
            }
        }
    }
}