package com.flybits.android.push.services

import android.app.PendingIntent
import android.content.Intent
import android.os.Handler
import android.os.Looper
import com.flybits.android.push.FlyNotification
import com.flybits.android.push.PushManager
import com.flybits.android.push.db.converters.PushDataConverter
import com.flybits.android.push.db.converters.PushDataToPushConverter
import com.flybits.android.push.models.newPush.DisplayablePush
import com.flybits.android.push.models.newPush.EventPush
import com.flybits.android.push.models.newPush.Push
import com.flybits.commons.library.SharedElementsFactory
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.logging.Logger
import com.google.firebase.messaging.FirebaseMessagingService
import com.google.firebase.messaging.RemoteMessage
import java.util.*

const val EXTRA_PUSH_NOTIFICATION = "com.flybits.android.push.services.push_notification"

/**
 * Extend this class to automate the handling of Flybits push notifications. Make sure to register the class
 * in the AndroidManifest <application> block, like so:
 *
 * <service
 *      android:name="name.of.the.class.extending.PushService">
 *      <intent-filter>
 *          <action android:name="com.google.firebase.MESSAGING_EVENT" />
 *      </intent-filter>
 * </service>
 *
 */
abstract class PushService: FirebaseMessagingService() {


    private val ERROR_CONVERSION_PUSHDATA_TO_PUSH = "Could Not Parse PushData into Push object. " +
            "This should not happen. Please contact support@flybits.com for more details"

    /**
     * This function will be invoked whenever a non Flybits push notification is received.
     *
     * @param remoteMessage The [RemoteMessage] associated with the non Flybits push notification.
     */
    abstract fun onNonFlybitsPushReceived(remoteMessage: RemoteMessage)

    /**
     *
     * @param push [Push] for which the icon will be displayed.
     *
     * @return id of the icon resource that will be displayed with notifications.
     */
    abstract fun getNotificationIconRes(push: DisplayablePush): Int

    /**
     *
     * Please Note : The Notification Channel creation is NOT required for devices with API less than 26,
     * notifications would be displayed even if non existing channel id is returned.
     * and
     * The Notification Channel creation is required for devices with API more than or equal to 26,
     * notifications would NOT be displayed if non existing channel id is returned.
     *
     * @param push The [Push] provided to extract information in order to construct the notification.
     *
     * @return id of already existing notification channel.
     */
    abstract fun getNotificationChannelId(push: Push): String

    override fun onMessageReceived(remoteMessage: RemoteMessage?) {
        super.onMessageReceived(remoteMessage)

        Logger.d("Push received: ${remoteMessage?.data}")

        if (remoteMessage != null) {
            try {

                val pushData = PushDataConverter.convertMapToPushData(remoteMessage.data)
                pushData?.let {

                    //We should consider saving the push here into the DB
                    val push = PushDataToPushConverter.convertPushDataToPush(it)
                    push?.let { itPush ->
                        when (itPush){
                            is DisplayablePush -> onDisplayableFlybitsPushReceived(itPush)
                            is EventPush -> onEntityFlybitsPushReceived(itPush)
                        }
                    } ?: Logger.e(ERROR_CONVERSION_PUSHDATA_TO_PUSH)
                } ?: onNonFlybitsPushReceived(remoteMessage)
            } catch (e: FlybitsException) {
                Logger.exception("ConciergeMessagingService.onMessageReceived()", e)
            }
        }
    }

    override fun onNewToken(token: String?) {
        super.onNewToken(token)
        if (token != null && SharedElementsFactory.get(applicationContext).getSavedJWTToken().isNotEmpty()) {
            PushManager.enablePush(applicationContext, token, HashMap(), null, Handler(Looper.getMainLooper()))
        }
    }

    /**
     * This function will be invoked right before a push notification is passed off to the
     * NotificationManager for display to the user. Its purpose is to allow the developer
     * to pass whatever data they need to the [Intent], and return a [PendingIntent]
     * that will be stored in the notification being sent.
     *
     * If you only need to change the [Intent] creation behaviour and not the [PendingIntent]
     * creation behaviour then override `onCreateIntent(push: Push)` instead.
     *
     * @param intent An already created intent that contains [Push] under the
     * ConciergeConstants.PUSH_EXTRA key, and by default uses the applications
     * launcher activity.
     *
     * @param push [Push] associated to the notification being shown to the user.
     */
    open fun onCreatePendingIntent(intent: Intent, push: Push): PendingIntent {
        return PendingIntent.getActivity(applicationContext
            , push.id.hashCode(), intent, PendingIntent.FLAG_UPDATE_CURRENT)
    }

    /**
     * This function will be invoked right before a push notification is passed off to the
     * NotificationManager for display to the user. Its purpose is to allow the developer
     * to pass whatever data they need to the [Intent] that will be stored in the [PendingIntent].
     *
     * To override the default [PendingIntent] override `onCreatePendingIntent(intent: Intent, push: Push)`
     *
     * @param push [Push] tied to the notification being currently created.
     *
     * @return [Intent] stored within the [PendingIntent] associated to the notification being
     * shown to the user.
     */
    open fun onCreateIntent(push: Push): Intent {
        val packageManager = applicationContext.getPackageManager()

        // package manager is provider of all the application information
        val mainIntent = Intent(Intent.ACTION_MAIN, null)
        mainIntent.addCategory(Intent.CATEGORY_LAUNCHER)

        val appList = packageManager.queryIntentActivities(mainIntent, 0)

        val className = appList.find {
            applicationContext.packageName == it.activityInfo.packageName
        }?.activityInfo?.name ?: throw ClassNotFoundException()

        return Intent(applicationContext, Class.forName(className)).apply {
            flags = Intent.FLAG_ACTIVITY_CLEAR_TOP or
                    Intent.FLAG_ACTIVITY_SINGLE_TOP or
                    Intent.FLAG_ACTIVITY_NEW_TASK
            putExtra(EXTRA_PUSH_NOTIFICATION, push)
        }
    }

    /**
     * This function will show the notification in the notification trey.
     *
     * @param push The [DisplayablePush] that is received from the Flybits server.
     * @param pendingIntent The [PendingIntent] that should be launched when the notification
     * selected.
     */
    open fun showNotification(push : DisplayablePush, pendingIntent : PendingIntent) {
        FlyNotification.Builder(
            applicationContext,
            push,
            getNotificationChannelId(push),
            getNotificationIconRes(push))
            .setPendingIntent(pendingIntent)
            .build()
            .show()
    }

    /**
     * This function will be invoked whenever a notification is received from the Flybits server
     * with the intention of being displayed to the end user. This is generally the case whenever
     * the [Push.action] is `PushAction.CUSTOM`.
     *
     * @param push [Push] received from Flybits
     */
    open fun onDisplayableFlybitsPushReceived(push: DisplayablePush) {
        val pendingIntent = this.onCreatePendingIntent(onCreateIntent(push), push)
        showNotification(push, pendingIntent)
    }

    /**
     * This function will be invoked whenever a Flybits push notification is received
     * indicating that a `PushEntity` has been either created (`PushAction.ADDED`) or
     * updated `PushAction.UPDATED` or deleted `PushAction.REMOVED`.
     *
     * @param push The [Push] which contains either the Create, Update or Delete `PushAction`.
     *
     */
    open fun onEntityFlybitsPushReceived(push: EventPush) {}
}