package com.flybits.android.push

import android.app.Notification
import android.app.NotificationManager
import android.app.PendingIntent
import android.content.Context
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.Canvas
import android.graphics.drawable.BitmapDrawable
import android.graphics.drawable.VectorDrawable
import android.os.Build
import android.support.annotation.ColorInt
import android.support.annotation.DrawableRes
import android.support.annotation.RequiresApi
import android.support.v4.app.NotificationCompat
import android.support.v4.content.ContextCompat
import com.flybits.android.push.analytics.PushAnalytics
import com.flybits.android.push.models.newPush.DisplayablePush
import java.util.*

/**
 * The purpose of this class is simplify the process of displaying a push notification in the
 * notification tray of the device. It reduces the complexities figuring out which feature are
 * available with the different versions of the Android OS. This class also provides an option
 * to retrieve the `NotificationCompat.Builder so that you can add additional options that might not be
 * implemented in the future that this class currently does not support.
 *
 * @param notificationManager [NotificationManager] to be used for showing the notification to the user.
 * @param id notification id
 * @param notification [NotificationCompat.Builder] containing the notification attributes
 * @param pushAnalytics [PushAnalytics] to be used in tracking analytics for the associated Flybits push. Leave
 * null if the notification hasn't came from Flybits.
 * @param flybitsPushRequestId The request id of the Flybits [Push], leave null if the notification hasn't
 * came from Flybits.
 *
 */
class FlyNotification internal constructor(
    private val notificationManager: NotificationManager,
    val id: String,
    val notification: Notification,
    private val pushAnalytics: PushAnalytics?,
    private val flybitsPushRequestId: String? = null
) {

    /**
     * Instantiate a [FlyNotification] using [Builder].
     *
     * @param builder [Builder] containing properties required to create the [FlyNotification].
     */
    internal constructor(builder: Builder): this(
        builder.context.getSystemService(Context.NOTIFICATION_SERVICE) as NotificationManager,
        builder.id,
        builder.notification.build(),
        PushAnalytics(builder.context),
        builder.flybitsPushRequestId
    )

    /**
     * Displays the notification within the Notification tray of the device and tracks analytics for the
     * associated [Push] if the notification came from Flybits.
     */
    fun show() {
        notificationManager.notify(id.hashCode(), notification)
        if (flybitsPushRequestId != null && pushAnalytics != null) {
            pushAnalytics.trackViewed(flybitsPushRequestId)
        }
    }

    override fun toString() = "{notificationId: $id, flybits push id: $flybitsPushRequestId, push analytics: $pushAnalytics}"

    override fun equals(other: Any?): Boolean {
        return other is FlyNotification && other.flybitsPushRequestId == flybitsPushRequestId && other.id == id
                && ((other.pushAnalytics != null && pushAnalytics != null)
                        || (other.pushAnalytics == null && pushAnalytics == null))
    }

    override fun hashCode(): Int {
        return Objects.hash(flybitsPushRequestId, id, pushAnalytics == null)
    }

    /**
     * The [Builder] class is responsible for building the [FlyNotification] object that is used to
     * display the Notification. This class handles any complexities in order to simplify the process of
     * building your Notification.
     *
     * @param context The context of the application.
     * @param id The unique identifier used to define the Notification.
     * @param channelId The `NotificationChannel` id that this notification should be assigned to.
     * @param resourceSmallIcon The resource that indicates which small icon should be displayed.
     * @param title The title to be display as part of the Notification. Null can be used for
     * this parameter.
     * @param body The body/message that is part of the Notification to be displayed. Null can
     * be used for this parameter.
     * @param flybitsPushRequestId The request id of the Flybits [Push] that triggered this notification.
     * Pass null if the notification has not came from Flybits. Find it at `Push.requestId`
     *
     */
    class Builder @JvmOverloads constructor(
        val context: Context,
        val id: String,
        channelId: String,
        @DrawableRes resourceSmallIcon: Int,
        val flybitsPushRequestId: String? = null
    ) {

        private var displayImageInBigStyle = false
        private var icon: Bitmap? = null
        private var title: String? = null
        private var body: String? = null

        val notification: NotificationCompat.Builder = NotificationCompat.Builder(
            context,
            channelId
        ).setSmallIcon(resourceSmallIcon)
        .setBadgeIconType(NotificationCompat.BADGE_ICON_SMALL)

        /**
         * Constructor that defines all the necessary components needed to build a Notification
         * using the Flybits [Push] object.
         *
         * @param context The context of the application.
         * @param push The parsed Flybits [Push] that should be displayed.
         * @param channelId The id of the `NotificationChannel` that the notification will be sent through.
         * @param resourceSmallIcon The resource that indicates which small icon should be displayed.
         */
        constructor(context: Context, push: DisplayablePush, channelId: String, @DrawableRes resourceSmallIcon: Int) : this(
            context,
            push.id,
            channelId,
            resourceSmallIcon,
            push.pushRequestId
        ) {
            setTitle(push.title)
            setBody(push.message)
        }

        /**
         * Set the title of the notification being displayed to the user.
         *
         * @param title being displayed to the user
         *
         * @return The [Builder] that can be passed to the [FlyNotification] class in order to
         * construct the appropriate notification.
         */
        fun setTitle(title: String?): Builder {
            title?.let{
                this.title = title
                notification.setContentTitle(title)
            }
            return this
        }

        /**
         * Set the body of the notification being displayed to the user.
         *
         * @param body being displayed to the user
         *
         * @return The [Builder] that can be passed to the [FlyNotification] class in order to
         * construct the appropriate notification.
         */
        fun setBody(body: String?): Builder {
            body?.let{
                this.body = body
                notification.setContentText(body)
            }
            return this
        }

        /**
         * Adds a list of `NotificationCompat.Action`s that should be displayed to the
         * end-user as part of the Notification.
         *
         * @param actions The list of `NotificationCompat.Action`s that can be added to the
         * notification.
         *
         * @return The [Builder] that can be passed to the [FlyNotification] class in order to
         * construct the appropriate notification.
         */
        fun addActions(vararg actions: NotificationCompat.Action): Builder {
            actions.forEach { notification.addAction(it) }
            return this
        }

        /**
         * Sets which icon to display as a badge for this notification.
         *
         * @param iconType Must be one of `NotificationCompat.BADGE_ICON_NONE`,
         * `NotificationCompat.BADGE_ICON_SMALL`, `NotificationCompat.BADGE_ICON_LARGE`
         *
         * @return The [Builder] that can be passed to the [FlyNotification] class in order to
         * construct the appropriate notification.
         */
        fun setBadgeIcon(@NotificationCompat.BadgeIconType iconType: Int): Builder {
            notification.setBadgeIconType(iconType)
            return this
        }

        /**
         * Set the large icon that is shown in the ticker and notification through a Bitmap.
         *
         * @param icon The Bitmap to be displayed.
         * @param displayImageInBigStyle true indicates that the Icon should be displayed in a Big
         * Style notification.
         *
         * @return The [Builder] that can be passed to the [FlyNotification] class in order to
         * construct the appropriate notification.
         */
        fun setLargeIcon(icon: Bitmap, displayImageInBigStyle: Boolean): Builder {
            notification.setLargeIcon(icon)
            this.icon = icon
            this.displayImageInBigStyle = displayImageInBigStyle
            return this
        }

        /**
         * Set the large icon that is shown in the ticker and notification through an application
         * resource.
         *
         * @param resourceIcon The resource used to construct the Icon to be displayed.
         * @param displayImageInBigStyle true indicates that the Icon should be displayed in a Big
         * Style notification.
         *
         * @return The [Builder] that can be passed to the [FlyNotification] class in order to
         * construct the appropriate notification.
         */
        @RequiresApi(Build.VERSION_CODES.LOLLIPOP)
        fun setLargeIcon(resourceIcon: Int, displayImageInBigStyle: Boolean): Builder {
            val bitmap = when (val drawable = ContextCompat.getDrawable(context, resourceIcon)) {
                is BitmapDrawable -> BitmapFactory.decodeResource(context.resources, resourceIcon)
                is VectorDrawable ->  {
                    val bitmap = Bitmap.createBitmap(
                        drawable.intrinsicWidth,
                        drawable.intrinsicHeight,
                        Bitmap.Config.ARGB_8888
                    )
                    val canvas = Canvas(bitmap)
                    drawable.setBounds(0, 0, canvas.width, canvas.height)
                    drawable.draw(canvas)
                    bitmap
                }
                else -> throw IllegalArgumentException("Invalid drawable id, ensure it is correct and that you are passing a BitmapDrawable or VectorDrawable")
            }
            setLargeIcon(bitmap, displayImageInBigStyle)
            return this
        }

        /**
         * Set the argb value that you would like the LED on the device to blink, as well as the
         * rate. The rate is specified in terms of the number of milliseconds to be on
         * and then the number of milliseconds to be off.
         *
         * @param argb Value that you would like the LED on the device to blink.
         * @param onMs Milliseconds rate that the lights should be on.
         * @param offMs Milliseconds rate that the lights should be off.
         *
         * @return The [Builder] that can be passed to the [FlyNotification] class in order to
         * construct the appropriate notification.
         */
        fun setLights(@ColorInt argb: Int, onMs: Int, offMs: Int): Builder {
            notification.setLights(argb, onMs, offMs)
            return this
        }

        /**
         * Supply a [PendingIntent] to send when the notification is clicked.
         *
         * @param intent The [PendingIntent] that should send when the notification is clicked.
         *
         * @return The [Builder] that can be passed to the [FlyNotification] class in order to
         * construct the appropriate notification.
         */
        fun setPendingIntent(intent: PendingIntent): Builder {
            notification.setContentIntent(intent)
            return this
        }

        /**
         * Set the vibration pattern to use.
         *
         * @param vibration The pattern to use for vibration once the notification is displayed.
         *
         * @return The [Builder] that can be passed to the [FlyNotification] class in order to
         * construct the appropriate notification.
         */
        fun setVibration(vibration: LongArray): Builder {
            notification.setVibrate(vibration)
            return this
        }

        /**
         * Create the [FlyNotification] based on all the options that have been set.
         *
         * @return The [FlyNotification] object that is used to display the notification.
         */
        fun build(): FlyNotification {
            if (icon != null) {
                if (displayImageInBigStyle) {
                    val bigPicture = NotificationCompat.BigPictureStyle()
                    bigPicture.bigPicture(icon).bigLargeIcon(null)
                    if (title != null) bigPicture.setBigContentTitle(title)
                    notification.setStyle(bigPicture)
                } else {
                    val bigText = NotificationCompat.BigTextStyle()
                    if (title != null) bigText.setBigContentTitle(title)
                    if (body != null) bigText.setSummaryText(body)
                    notification.setStyle(bigText)
                }
            }

            return FlyNotification(this)
        }
    }
}
