package com.flybits.android.push.models;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.os.Parcel;
import android.os.Parcelable;
import android.support.annotation.NonNull;

import com.flybits.android.push.PushScope;
import com.flybits.android.push.deserializations.DeserializePushToken;
import com.flybits.commons.library.api.FlyAway;
import com.flybits.commons.library.api.results.ObjectResult;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.models.internal.Result;

import java.util.HashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * This class is used to define a FCM push token which should be transferred from the device, once
 * it is obtained from Google, to the Flybits Push Server. Through this {@code PushToken} class the
 * Flybits Push Server will be able to know to which {@code User} to send Push notification to
 * based on specific Server and Business logic.
 *
 * @since 2.1.8
 */
public class PushToken implements Parcelable {

    public static final String  API                = PushScope.ROOT + "/token";

    /**
     * The properties that are associated to this device token.
     */
    private HashMap<String, String> data;

    /**
     * The FCM/APNS {@code PushToken}.
     */
    private String deviceToken;

    /**
     * Indicates through which service the {@code PushToken} was generated from.
     */
    private long network;

    /**
     * Constructor used to define the token obtained from Google's FCM service.
     *
     * @param token The FCM token obtained from Google.
     */
    public PushToken(@NonNull String token){
        deviceToken = token;

        //1 is for FCM
        network = 1;
    }

    /**
     * Get additional data that is associated to the {@code PushToken}. This data is added by an
     * application as additional metadata.
     *
     * @return The {@code HashMap} containing additional information associated to the
     * {@code PushToken}.
     */
    public HashMap<String, String> getData() {
        return data;
    }

    /**
     * Get the FCM token associated to the device which is retrieved from a Google server.
     *
     * @return The FCM token associated to this device.
     */
    public String getDeviceToken() {
        return deviceToken;
    }

    /**
     * Get the type of token that is associated to the device.
     *
     * @return The integer value that indicates which type of device this token represents, 1 is for
     * Google.
     */
    public long getType() {
        return network;
    }

    /**
     * Constructor used to define the token obtained from Google's FCM service.
     *
     * @param token The FCM token obtained from Google.
     * @param data Key-Value pairs that is associated to this specific instance of Push
     *             Notifications. These Key-Value pair allow clients to define specific
     *             characteristics associated to their Flybits instance. In many cases this is
     *             useful for clients to require the Flybits Push system to notify the client's own
     *             Push Notification system from where their notification will initiate.
     */
    public PushToken(String token, HashMap<String, String> data){
        deviceToken = token;
        network     = 1;
        this.data   = data;
    }

    /**
     * Constructor used for un-flattening a {@code Push} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code Push} parcel.
     */
    protected PushToken(Parcel in){
        deviceToken = in.readString();
        network     = in.readLong();
    }

    /**
     * Describe the kinds of special objects contained in this Parcelable's marshalled representation.
     *
     * @return a bitmask indicating the set of special object types marshalled by the Parcelable.
     */
    public int describeContents() {
        return 0;
    }


    /**
     * Flatten this {@code PushToken} into a Parcel.
     *
     * @param out The Parcel in which the {@code PushToken} object should be written.
     * @param flags Additional flags about how the DateOfBirth object should be written.
     * May be 0 or {@link #PARCELABLE_WRITE_RETURN_VALUE}.
     */
    public void writeToParcel(Parcel out, int flags) {
        out.writeString(deviceToken);
        out.writeLong(network);
    }

    /**
     * Parcelable.Creator that instantiates {@code PushToken} objects
     */
    public static final Parcelable.Creator<PushToken> CREATOR = new Parcelable.Creator<PushToken>() {
        public PushToken createFromParcel(Parcel in) {
            return new PushToken(in);
        }

        public PushToken[] newArray(int size) {
            return new PushToken[size];
        }
    };

    /**
     * Get the FCM Token that is associated to the logged in {@code User}.
     *
     * @param context The context of the activity that is retrieving the FCM Push Token.
     * @param callback The callback that indicates whether or not the GET request was successful or
     *                 it failed.
     * @return The {@code ObjectResult} that returns the Push Token within the {@code onSuccess()}
     * method of the {@code callback}.
     */
    public static ObjectResult<PushToken> get(@NonNull final Context context, final ObjectResultCallback<PushToken> callback) {

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ObjectResult<PushToken> query = new ObjectResult<PushToken>(callback, handler, executorService);

        executorService.execute(new Runnable() {
            public void run() {
                try {
                    final Result<PushToken> result = FlyAway.get(context, API, new DeserializePushToken(), "PushToken.get", PushToken.class);
                    query.setResult(result);
                } catch (final FlybitsException e) {
                    query.setFailed(e);
                }
            }
        });
        return query;
    }
}
