package com.flybits.android.push.models;

import android.arch.persistence.room.ColumnInfo;
import android.arch.persistence.room.Entity;
import android.arch.persistence.room.Ignore;
import android.arch.persistence.room.PrimaryKey;
import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.os.Parcel;
import android.os.Parcelable;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.flybits.android.push.PushScope;
import com.flybits.android.push.db.PushDatabase;
import com.flybits.android.push.db.caching.PushCacheLoader;
import com.flybits.android.push.deserializations.DeserializePushNotification;
import com.flybits.android.push.deserializations.DeserializePushNotificationCustomFields;
import com.flybits.android.push.models.results.PushResult;
import com.flybits.android.push.utils.PushQueryParameters;
import com.flybits.commons.library.api.FlyAway;
import com.flybits.commons.library.api.results.BasicResult;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback;
import com.flybits.commons.library.deserializations.DeserializePagedResponse;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.models.internal.PagedResponse;
import com.flybits.commons.library.models.internal.Result;
import com.flybits.internal.db.CommonsDatabase;
import com.flybits.internal.db.models.CachingEntry;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * This class is used to represent a push notification obtained from the Flybits Push Server through
 * either FCM or MQTT depending whether the notification was sent as either a foreground or
 * background notification.
 *
 * <p>A push notification can either be represented through a {@link android.app.Notification} found
 * within the OS' notification tray.
 */
@Entity(tableName = "push")
public class Push implements Parcelable {

    static final            String API                              = PushScope.ROOT + "/notifications";

    //Db schema
    public static final     String COLUMN_DATABASE_ID               = "databaseId";
    public static final     String COLUMN_ACTION                    = "action";
    public static final     String COLUMN_CUSTOM_FIELDS             = "customFields";
    public static final     String COLUMN_CUSTOM_FIELDS_AS_STRING   = "customFieldsAsString";
    public static final     String COLUMN_CATEGORY                  = "category";
    public static final     String COLUMN_ENTITY                    = "entity";
    public static final     String COLUMN_ID                        = "id";
    public static final     String COLUMN_PUSH_REQUEST_ID           = "pushRequestId";
    public static final     String COLUMN_MESSAGE                   = "message";
    public static final     String COLUMN_TIMESTAMP                 = "timestamp";
    public static final     String COLUMN_TITLE                     = "title";
    public static final     String COLUMN_VERSION                   = "version";
    public static final     String COLUMN_META_DATA_ID              = "metadataID";

    @PrimaryKey
    @NonNull
    @ColumnInfo(name = COLUMN_DATABASE_ID)
    private String databaseId;

    @ColumnInfo(name = COLUMN_ACTION)
    @NonNull
    private PushAction action;

    @ColumnInfo(name = COLUMN_CUSTOM_FIELDS)
    private HashMap<String, String> customFields;

    @ColumnInfo(name = COLUMN_CUSTOM_FIELDS_AS_STRING)
    private String customFieldsAsString;

    @ColumnInfo(name = COLUMN_CATEGORY)
    @NonNull
    private PushCategory category;

    @ColumnInfo(name = COLUMN_ENTITY)
    private PushEntity entity;

    @ColumnInfo(name = COLUMN_ID)
    @NonNull
    private String id;

    @ColumnInfo(name = COLUMN_PUSH_REQUEST_ID)
    @NonNull
    private String requestId;

    @ColumnInfo(name = COLUMN_MESSAGE)
    private String message;

    @ColumnInfo(name = COLUMN_TIMESTAMP)
    private long timestamp;

    @ColumnInfo(name = COLUMN_TITLE)
    private String title;

    @ColumnInfo(name = COLUMN_VERSION)
    private long version;

    @ColumnInfo(name = COLUMN_META_DATA_ID)
    private String metadataID;

    /**
     * Constructor used to initiate the {@code Push} object.
     *
     * @param id The unique identifier for this {@code Push} object.
     * @param entity The {@link PushEntity} that indicates what the push notification is about.
     * @param action The {@link PushAction} that should be taken on the {@link PushEntity}.
     * @param category The {@link PushCategory} of push notification this is.
     * @param version The version of the Push notification. Based on the version the SDK will know
     *                how parse the push notification.
     * @param timestamp The timestamp, in milliseconds, representing when the push notification was
     *                  created.
     * @param title The title of an alert notification.
     * @param message The message of an alert notification.
     */
    @Ignore
    public Push(@NonNull String id, @NonNull String entity, @NonNull String action, @NonNull String category, @NonNull String requestId, long version, long timestamp,
                String title, String message) {

        this.id                 = id;
        this.entity             = PushEntity.fromKey(entity);
        this.action             = PushAction.fromKey(action);
        this.category           = PushCategory.fromKey(category);
        this.version            = version;
        this.timestamp          = timestamp;
        this.title              = title;
        this.message            = message;
        this.customFields       = new HashMap<>();
        this.databaseId         = id+timestamp;
        this.requestId          = requestId;
    }

    /**
     * Constructor used to initiate the {@code Push} object.
     *
     * @param id The unique identifier for this {@code Push} object.
     * @param entity The {@link PushEntity} that indicates what the push notification is about.
     * @param action The {@link PushAction} that should be taken on the {@link PushEntity}.
     * @param category The {@link PushCategory} of push notification this is.
     * @param version The version of the Push notification. Based on the version the SDK will know
     *                how parse the push notification.
     * @param timestamp The timestamp, in milliseconds, representing when the push notification was
     *                  created.
     * @param title The title of an alert notification.
     * @param message The message of an alert notification.
     * @param customFieldsAsString The body, represented by a String, of the content of the push notification.
     * @param metadataID The unique identifier that represents the structure of the metadata.
     */
    @Ignore
    public Push(@NonNull String id, @NonNull String entity, @NonNull String action, @NonNull String category, @NonNull String requestId,
                long version, long timestamp, String title, String message, String customFieldsAsString, String metadataID) {

        this(id, entity, action, category, requestId, version, timestamp, title, message);
        this.customFieldsAsString   = customFieldsAsString;
        this.metadataID             = metadataID;

        DeserializePushNotificationCustomFields deserializer    = new DeserializePushNotificationCustomFields();
        if ( deserializer.fromJson(this.customFieldsAsString) != null){
            this.customFields   =deserializer.fromJson(this.customFieldsAsString);
        }
    }

    /**
     * Constructor used to initiate the {@code Push} object.
     *
     * @param id The unique identifier for this {@code Push} object.
     * @param entity The {@link PushEntity} that indicates what the push notification is about.
     * @param action The {@link PushAction} that should be taken on the {@link PushEntity}.
     * @param category The {@link PushCategory} of push notification this is.
     * @param version The version of the Push notification. Based on the version the SDK will know
     *                how parse the push notification.
     * @param timestamp The timestamp, in milliseconds, representing when the push notification was
     *                  created.
     * @param title The title of an alert notification.
     * @param message The message of an alert notification.
     * @param customFieldsAsString The body, represented by a String, of the content of the push notification.
     * @param metadataID The unique identifier that represents the structure of the metadata.
     */
    public Push(@NonNull String id, @NonNull PushEntity entity, @NonNull PushAction action, @NonNull PushCategory category, @NonNull String requestId,
                long version, long timestamp, String title, String message, String customFieldsAsString, String metadataID) {

        this(id, entity.getKey(), action.getKey(), category.getKey(), requestId, version, timestamp, title, message);
        this.customFieldsAsString   = customFieldsAsString;
        this.metadataID             = metadataID;

        DeserializePushNotificationCustomFields deserializer    = new DeserializePushNotificationCustomFields();
        if ( deserializer.fromJson(this.customFieldsAsString) != null){
            this.customFields   =deserializer.fromJson(this.customFieldsAsString);
        }
    }

    /**
     * Constructor used for un-flattening a {@code Push} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code Push} parcel.
     */
    @Ignore
    public Push(Parcel in) {
        entity                  = PushEntity.fromKey(in.readString());
        action                  = PushAction.fromKey(in.readString());
        category                = PushCategory.fromKey(in.readString());
        version                 = in.readLong();
        timestamp               = in.readLong();
        customFieldsAsString    = in.readString();
        message                 = in.readString();
        title                   = in.readString();
        id                      = in.readString();
        metadataID              = in.readString();
        databaseId              = in.readString();
        requestId               = in.readString();

        DeserializePushNotificationCustomFields deserializeResult = new DeserializePushNotificationCustomFields();
        customFields = deserializeResult.fromJson(customFieldsAsString);
    }

    /**
     * Get the {@link PushAction} that has occurred on the {@code entity}.
     *
     * @return The {@link PushAction} of the received {@code entity}.
     */
    @NonNull
    public PushAction getAction(){
        return action;
    }

    /**
     * Get the {@link PushCategory} that the {@code entity} belongs to.
     *
     * @return The {@link PushCategory} of the received {@code entity}.
     */
    @NonNull
    public PushCategory getCategory(){
        return category;
    }

    /**
     * Get the custom field object represented as a String in case the developer would like to
     * perform their own deserializations. This information is set through the custom fields portion
     * of the push notification. It may be null in the event that no custom fields were set.
     *
     * @return The String representation of the {@code Push}'s custom fields.
     */
    public String getCustomFieldsAsString(){
        return customFieldsAsString;
    }

    /**
     * Get a HashMap of custom fields that have been set within the Experience Studio. The custom
     * fields are represented with a key-value {@code HashMap}.
     *
     * @return The {@code HashMap} that contains Key-Value pairs of the Custom Fields object
     * assigned in the Experience Studio.
     */
    public HashMap<String, String> getCustomFields(){
        return customFields;
    }

    /**
     * Get the {@link PushEntity} that the {@code entity} belongs to.
     *
     * @return The {@link PushEntity} of the received {@code entity}.
     */
    @NonNull
    public PushEntity getEntity(){
        return entity;
    }

    /**
     * The unique identifier representing this push notification. This is unique for the push
     * notification across the ecosystem.
     *
     * @return The unique identifier of the {@code Push} Notification. This identifier can be used
     * later on to retrieve/delete content associated to this {@code Push} notification.
     */
    @NonNull
    public String getId(){
        return id;
    }

    /**
     * Get the message of an notification. Alert notification should be displayed within the
     * notification bar notification's header field. In silent notifications this field will be
     * null.
     *
     * @return The message of the push notification.
     */
    public String getMessage(){
        return message;
    }

    /**
     * Get the unique identifier of the metadata structure that is part of the custom fields.
     *
     * @return The GUID that represents the unique identifier of the custom fields.
     */
    @Nullable
    public String getMetadataID(){
        return metadataID;
    }

    /**
     * Get the timestamp which indicates when the push notification was generated.
     *
     * @return The epoch time of when the push notification was generated.
     */
    public long getTimestamp(){
        return timestamp;
    }

    /**
     * Get the title of an Push/MQTT notification. The notification should be displayed within the
     * notification bar notification's title field. In silent notifications this field will be
     * null.
     *
     * @return The title of the push notification.
     */
    public String getTitle(){
        return title;
    }

    /**
     * Get the version of the {@code Push} notification. This version is used to identify the push
     * notification structure and can be used to parse information based on this.
     *
     * @return The version number of the Push structure.
     */
    public long getVersion(){
        return version;
    }

    /**
     * Get the unique id for this specific instance of Push. This is a temporary addition
     * since the id associated with Push isn't unique when returned from the server causing
     * only one instance of the same request to be stored in the database.
     *
     * @return unique id.
     */
    @NonNull
    public String getDatabaseId() {
        return databaseId;
    }

    /**
     * Get the request id. This id isn't unique, meaning different instances of {@link Push}
     * may be associated the same request id. For a unique id please look at databaseId.
     * Request id represents the unique identification for the structure of the push and its contents but
     * not the timestamp.
     *
     * @return the request id.
     */
    @NonNull
    public String getRequestId() {
        return requestId;
    }

    /**
     * Sets the {@link PushAction} that triggered the {@link Push} notification to be sent.
     *
     * @param action The {@link PushAction} that triggered the {@link Push} notification to be sent.
     */
    public void setAction(@NonNull PushAction action) {
        this.action = action;
    }

    /**
     * Set the custom field object represented as a String in case the developer would like to
     * perform their own deserializations. This information is set through the custom fields portion
     * of the push notification. It may be null in the event that no custom fields were set.
     *
     * @param customFieldsAsString The String representation of the {@code Push}'s custom fields.
     */
    public void setCustomFieldsAsString(String customFieldsAsString) {
        this.customFieldsAsString = customFieldsAsString;
    }

    /**
     * Sets the {@link PushCategory} of the {@link Push} notification which indicates which
     * catalogue the {@link Push} notification is associated to.
     *
     * @param category The {@link PushCategory} of the {@link Push} notification.
     */
    public void setCategory(@NonNull PushCategory category) {
        this.category = category;
    }

    /**
     * Set a HashMap of custom fields that have been set within the Experience Studio. The custom
     * fields are represented as a key-value {@code HashMap}.
     *
     * @param customFields The {@code HashMap} that contains Key-Value pairs of the Custom Fields object
     * assigned in the Experience Studio.
     */
    public void setCustomFields(HashMap<String, String> customFields) {
        this.customFields = customFields;
    }

    /**
     * Set the {@link PushEntity} of the which entity the {@link Push} notification is associated
     * to.
     *
     * @param entity The {@link PushEntity} of the received {@code entity}.
     */
    public void setEntity(@NonNull PushEntity entity) {
        this.entity = entity;
    }

    /**
     * Set the unique identifier for the {@code Push} notification. This identifier is used to
     * delete the {@code Push} as well as retrieve information that may be associated to it.
     *
     * @param id The unique GUID that represents this {@code Push}.
     */
    public void setId(@NonNull String id) {
        this.id = id;
    }

    /**
     * Set the message of an notification. Alert notification should be displayed within the
     * notification bar notification's header field. In silent notifications this field will be
     * null.
     *
     * @param message The message of the push notification.
     */
    public void setMessage(String message) {
        this.message = message;
    }

    /**
     * Set the metadata identifier of the custom field structure.
     *
     * @param metadataID The unique identifier that represents the structure of the metadata.
     */
    public void setMetadataID(String metadataID) {
        this.metadataID = metadataID;
    }

    /**
     * Set the timestamp which indicates when the push notification was generated.
     *
     * @param timestamp The epoch time of when the push notification was generated.
     */
    public void setTimestamp(long timestamp) {
        this.timestamp = timestamp;
    }

    /**
     * Get the title of an Push/MQTT notification. The notification should be displayed within the
     * notification bar notification's title field. In silent notifications this field will be
     * null.
     *
     * @param title The title of the push notification.
     */
    public void setTitle(String title) {
        this.title = title;
    }

    /**
     * Set the version of the {@code Push} notification. This version is used to identify the push
     * notification structure and can be used to parse information based on this.
     *
     * @param version The version number of the Push structure.
     */
    public void setVersion(long version) {
        this.version = version;
    }

     /** Get the unique id for this specific instance of {@link Push}. This is a temporary addition
     * since the id associated with Push isn't unique when returned from the server causing
     * only one instance of the same request to be stored in the database.
     *
     * @param databaseId The unique id for this specific instance of {@link Push}
     */
    public void setDatabaseId(@NonNull String databaseId) {
        this.databaseId = databaseId;
    }

    /**
     * Set the request id.
     *
     * @param requestId The request id.
     */
    public void setRequestId(@NonNull String requestId) {
        this.requestId = requestId;
    }

    @Override
    public String toString() {
        return  "{ \"id\" : " + id +
                ", \"version\" : " + version +
                ", \"timestamp\" : " + timestamp +
                ", \"entity\" : \"" + entity.getKey()+"\""+
                ", \"action\" : \"" + action.getKey()+"\""+
                ", \"category\" : \"" + category.getKey()+"\""+
                ", \"title\" : \"" + title+"\""+
                ", \"alert\" : \"" + message+"\""+
                ", \"body\" : " + customFieldsAsString +
                ", \"metadataID\" : " + metadataID +
                '}';
    }

    /**
     * Describe the kinds of special objects contained in this Parcelable's marshalled representation.
     *
     * @return a bitmask indicating the set of special object types marshalled by the Parcelable.
     */
    public int describeContents() {
        return 0;
    }

    /**
     * Flatten this {@code Push} into a Parcel.
     *
     * @param out The Parcel in which the {@code Push} object should be written.
     * @param flags Additional flags about how the DateOfBirth object should be written.
     * May be 0 or {@link #PARCELABLE_WRITE_RETURN_VALUE}.
     */
    public void writeToParcel(Parcel out, int flags) {
        out.writeString(entity.getKey());
        out.writeString(action.getKey());
        out.writeString(category.getKey());
        out.writeLong(version);
        out.writeLong(timestamp);
        out.writeString(customFieldsAsString);
        out.writeString(message);
        out.writeString(title);
        out.writeString(id);
        out.writeString(metadataID);
        out.writeString(databaseId);
        out.writeString(requestId);
    }

    /**
     * Parcelable.Creator that instantiates {@code Push} objects
     */
    public static final Parcelable.Creator<Push> CREATOR = new Parcelable.Creator<Push>() {
        public Push createFromParcel(Parcel in) {
            return new Push(in);
        }

        public Push[] newArray(int size) {
            return new Push[size];
        }
    };

    /**
     * Retrieve all the {@link com.flybits.android.push.models.Push} notifications that have been
     * sent to the connected user. It is important to make sure that the user is in fact connected
     * to Flybits. This can be achieved through the {@code FlybitsManager#connect()} function.
     *
     * @param context The state of the application.
     * @param params The {@link PushQueryParameters} which allow you to easily define the pagination
     *                schema for the request you are trying to make.
     * @param callback The callback that initiated when the request is completed. It will contain
     *                 either a successful method or failure with a
     *                 {@code FlybitsException} which indicates the reason for failure.
     * @return The {@link ExecutorService} which can be shutdown by the application developer in
     * case the request is taking too long or the user no longer needs the result from the request.
     */
    public static PushResult get(@NonNull final Context context, @NonNull final PushQueryParameters params,
                                 final PagedResultCallback<Push> callback) {

        Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final PushResult query = new PushResult(context, callback, executorService, params, handler);
        query.setService(executorService);

        executorService.execute(new Runnable() {
            public void run() {

                try {
                    DeserializePushNotification singleDersializer   = new DeserializePushNotification();
                    final Result<PagedResponse<Push>> result = FlyAway.get(context, API, params, new DeserializePagedResponse<Push>(singleDersializer), "Push.get");

                    List<String> listOfIds = null;

                    //Update the DB for caching/observing purposes
                    if (result.getStatus() == RequestStatus.COMPLETED) {
                        if ((params.getQueryParams().get("offset") == null
                                || params.getQueryParams().get("offset").size() == 0
                                || params.getQueryParams().get("offset").get(0).equals("0"))
                                && (params.getCachingKey() != null)) {

                            listOfIds = CommonsDatabase.getDatabase(context).cachingEntryDAO().getUniqueIdsByCachingKey(params.getCachingKey());
                        }

                        int maxNumberOfRecordsToSave = (params.getCachingLimit() < result.getResult().getItems().size() + result.getResult().getPagination().getOffset())
                                ? params.getCachingLimit() : result.getResult().getItems().size();

                        if (params.getCachingKey() != null && maxNumberOfRecordsToSave > 0) {
                            ArrayList<CachingEntry> entries = new ArrayList<>();
                            ArrayList<Push> pushesToCache = new ArrayList<>();
                            for (int i = 0; i < result.getResult().getItems().size(); i++) {
                                if (i >= maxNumberOfRecordsToSave){
                                    break;
                                }

                                pushesToCache.add(result.getResult().getItems().get(i));
                                entries.add(new CachingEntry(PushCacheLoader.PUSH_CACHE_KEY, result.getResult().getItems().get(i).getId()));
                            }

                            if (listOfIds == null){
                                //pulling initial content so remove all content for caching entry
                                PushDatabase.getDatabase(context).pushDao().deleteByIds(listOfIds);
                                CommonsDatabase.getDatabase(context).cachingEntryDAO().deleteAllByCachingKey(params.getCachingKey());
                            }
                            PushDatabase.getDatabase(context).pushDao().insert(pushesToCache);
                            CommonsDatabase.getDatabase(context).cachingEntryDAO().insert(entries);
                        }
                    }
                    query.setResult(result, params);
                } catch (final FlybitsException e) {
                    query.setFailed(e);
                }
            }
        });
        return query;
    }

    /**
     * Retrieve all the {@link com.flybits.android.push.models.Push} notifications that have been
     * sent to the connected user. It is important to make sure that the user is in fact connected
     * to Flybits. This can be achieved through the {@code FlybitsManager#connect()} function.
     *
     * @param mContext The state of the application.
     * @param params The {@link PushQueryParameters} which allow you to easily define the pagination
     *                schema for the request you are trying to make.
     * @return The {@link ExecutorService} which can be shutdown by the application developer in
     * case the request is taking too long or the user no longer needs the result from the request.
     */
    public static PushResult get(@NonNull final Context mContext, @NonNull final PushQueryParameters params) {
       return get(mContext, params, null);
    }

    /**
     * Deletes a specific {@link com.flybits.android.push.models.Push} notification from the list of
     * notifications sent to the connected user. It is important to make sure that the user is in
     * fact connected to Flybits. This can be achieved through the {@code FlybitsManager#connect()}
     * function.
     *
     * @param mContext The state of the application.
     * @param callback The callback that initiated when the request is completed. It will contain
     *                 either a successful method or failure with a
     *                 {@code FlybitsException} which indicates the reason for failure.
     * @return The {@link ExecutorService} which can be shutdown by the application developer in
     * case the request is taking too long or the user no longer needs the result from the request.
     */
    public BasicResult delete(@NonNull final Context mContext, @NonNull final BasicResultCallback callback){

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult resultObject    = new BasicResult(mContext, callback, executorService, handler);
        executorService.execute(new Runnable() {
            public void run() {

                try {
                    final Result result = FlyAway.delete(mContext, API, "Push.delete", id);

                    if (result.getStatus() == RequestStatus.COMPLETED){
                        PushDatabase.getDatabase(mContext).pushDao().delete(Push.this);
                    }
                    resultObject.setResult(result);
                }catch(final FlybitsException e){
                    resultObject.setFailed(e);
                }
            }
        });
        return resultObject;
    }
}
