package com.flybits.android.push.api;

import android.content.Context;
import android.content.Intent;
import android.support.annotation.NonNull;
import android.support.v4.content.LocalBroadcastManager;

import com.flybits.android.push.db.PushDatabase;
import com.flybits.android.push.deserializations.DeserializePushNotification;
import com.flybits.android.push.exceptions.FlybitsPushException;
import com.flybits.android.push.models.Push;
import com.flybits.commons.library.deserializations.IDeserializer;

import org.json.JSONException;
import org.json.JSONObject;

import java.math.BigDecimal;
import java.util.Map;

import static com.flybits.android.push.FlyingPushConstants.ACTION_PUSH_CUSTOM;
import static com.flybits.android.push.FlyingPushConstants.EXTRA_PUSH_NOTIFICATION;
import static com.flybits.android.push.FlyingPushConstants.INTERNAL_ACTION_PUSH_CONTEXT;
import static com.flybits.android.push.FlyingPushConstants.INTERNAL_ACTION_PUSH_KERNEL;

public class FlyPushParsing {

    public static Push parsePushNotification(@NonNull final Context context, @NonNull Map bundle) throws FlybitsPushException {
        verifyPushNotificationIsFlybits(bundle);

        Push notification        = buildPushNotification(bundle);
        if (notification != null) {
            Runnable r = new InsertIntoDB(context, notification);
            new Thread(r).start();
            broadcastPush(context, notification);
        }
        return notification;
    }

    public static void parseMQTTNotification(final Context context, String json) throws IllegalArgumentException {
        Push deserializedPushInfo = new DeserializePushNotification().fromJson(json);
        broadcastPush(context, deserializedPushInfo);
    }

    protected static void verifyPushNotificationIsFlybits(Map map) throws FlybitsPushException{

        if (map == null || !map.containsKey("category") || !map.containsKey("version") || !map.containsKey("entity")){
            throw new FlybitsPushException("An invalid FCM bundle was provided. Make sure that the FCM Bundle follows the Flybits Push Model.");
        }
    }

    protected static Push buildPushNotification(@NonNull Map map){
        long timestamp = -1;
        if (map.containsKey("timestamp")){
            BigDecimal bd = new BigDecimal(map.get("timestamp").toString());
            timestamp = bd.longValue();
        }
        long version        = (map.containsKey("version")) ? Long.parseLong(map.get("version").toString()) : -1 ;
        String body         = (map.containsKey("body")) ? map.get("body").toString() : null;
        String metadataID   = "unknownID";

        if((map.containsKey("alert") && map.get("alert").toString().length() != 0) || (map.containsKey("title") && map.get("title").toString().length() != 0)) {

            String alert = ((map.containsKey("alert") && map.get("alert").toString().length() != 0)) ? map.get("alert").toString() : null;
            String title = ((map.containsKey("title") && map.get("title").toString().length() != 0)) ? map.get("title").toString() : null;

            if (body != null){

                try {
                    JSONObject customFieldObject = new JSONObject(body);
                    if (!customFieldObject.isNull("pushPayloadTypeId")){
                        metadataID   = customFieldObject.getString("pushPayloadTypeId");
                    }

                    return new Push(map.get("id").toString(), map.get("entity").toString(), map.get("action").toString(),
                            map.get("category").toString(), version, timestamp, title, alert, body, metadataID);
                }catch (JSONException ex){}
            }

            return new Push(map.get("id").toString(), map.get("entity").toString(), map.get("action").toString(),
                    map.get("category").toString(), version, timestamp, title, alert, body, metadataID);
        }

        return  null;
    }

    private static void broadcastPush(final Context context, Push push){

        switch (push.getCategory()){
            case CONTEXT:
                decodePushContext(context, push);
                break;
            case KERNEL:
                decodePushKernel(context, push);
                break;
            case CUSTOM:
                decodePushCustom(context, push);
                break;
        }
    }

    private static <T extends IDeserializer> void decodePushContext(Context context, Push deserializedPushInfo){
        Intent intentCustom = new Intent(INTERNAL_ACTION_PUSH_CONTEXT);
        intentCustom.putExtra(EXTRA_PUSH_NOTIFICATION, deserializedPushInfo);
        LocalBroadcastManager.getInstance(context).sendBroadcast(intentCustom);
    }


    private static <T extends IDeserializer> void decodePushKernel(Context context, Push deserializedPushInfo){
        Intent intentCustom = new Intent(INTERNAL_ACTION_PUSH_KERNEL);
        intentCustom.putExtra(EXTRA_PUSH_NOTIFICATION, deserializedPushInfo);
        LocalBroadcastManager.getInstance(context).sendBroadcast(intentCustom);
    }

    private static <T extends IDeserializer> void decodePushCustom(Context context, Push deserializedPushInfo ){
        Intent intentCustom = new Intent(ACTION_PUSH_CUSTOM);
        intentCustom.putExtra(EXTRA_PUSH_NOTIFICATION, deserializedPushInfo);
        LocalBroadcastManager.getInstance(context).sendBroadcast(intentCustom);
    }

    private static class InsertIntoDB implements Runnable {

        private Push notification;
        private Context context;

        public InsertIntoDB(Context context, Push notification) {
            this.notification   = notification;
            this.context        = context;
        }

        public void run() {
            PushDatabase.getDatabase(context).pushDao().insert(notification);
        }
    }
}
