package com.flybits.android.push.services;

import android.content.ComponentName;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.os.Bundle;
import android.support.v4.content.LocalBroadcastManager;
import com.flybits.android.push.FlybitsNotificationManager;
import com.flybits.android.push.PushManager;
import com.flybits.android.push.R;
import com.flybits.android.push.exceptions.FlybitsPushException;
import com.flybits.android.push.models.Push;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.logging.Logger;
import com.google.firebase.messaging.FirebaseMessagingService;
import com.google.firebase.messaging.RemoteMessage;

import java.util.Map;

import static com.flybits.android.push.FlyingPushConstants.ACTION_PUSH_RECEIVED_FCM;
import static com.flybits.android.push.FlyingPushConstants.ACTION_PUSH_RECEIVED_FLYBITS;
import static com.flybits.android.push.FlyingPushConstants.EXTRA_PUSH_NOTIFICATION;

public class FlybitsPushService extends FirebaseMessagingService {

    private final String _TAG = "FlybitsFirebaseReceiver";

    @Override
    public void onMessageReceived(final RemoteMessage message){

        Logger.setTag(_TAG).i("FlybitsFCMReceiver->Received Message: " + message.getData());
        Map data = message.getData();
        PushManager.parsePushNotification(getBaseContext(), data, new ObjectResultCallback<Push>() {
            @Override
            public void onSuccess(Push push) {
                Logger.setTag(_TAG).i("Received Flybits Push: " + push.toString());

                FlybitsNotificationManager builder   = new FlybitsNotificationManager.Simplifier(
                        getApplicationContext(), push.getId(), push.getTitle(), push.getMessage(), getResource())
                        .build();
                builder.show();

                Intent intentCustom = new Intent(ACTION_PUSH_RECEIVED_FLYBITS);
                intentCustom.putExtra(ACTION_PUSH_RECEIVED_FLYBITS, push);
                LocalBroadcastManager.getInstance(getBaseContext()).sendBroadcast(intentCustom);
            }

            @Override
            public void onException(FlybitsException e) {
                if (e instanceof FlybitsPushException) {
                    if (message.getNotification() != null) {
                        Logger.setTag(_TAG).i("Received FCM Push: " + message.getNotification().getBody());
                        FlybitsNotificationManager builder   = new FlybitsNotificationManager.Simplifier(
                                getApplicationContext(), message.getMessageId(), message.getNotification().getTitle(), message.getNotification().getBody(), getResource())
                                .build();
                        builder.show();
                    }

                    Intent intentCustom = new Intent(ACTION_PUSH_RECEIVED_FCM);
                    intentCustom.putExtra(EXTRA_PUSH_NOTIFICATION, message);
                    LocalBroadcastManager.getInstance(getBaseContext()).sendBroadcast(intentCustom);
                }
            }
        });
    }

    private int getResource(){
        try {
            ComponentName myService = new ComponentName(this, this.getClass());
            Bundle bundle = getPackageManager().getServiceInfo(myService, PackageManager.GET_META_DATA).metaData;

            if (bundle.containsKey("com.flybits.push.notification.icon")) {
                return  bundle.getInt("com.flybits.push.notification.icon");
            }
        }catch (PackageManager.NameNotFoundException | NullPointerException e) {
            Logger.exception("FlybitsPushService.getResource", e);
        }
        Logger.setTag(_TAG).w("It appears that you have not set your icon's resource field within your AndroidManifest declaration of the FlybitsPushService. Please visit https://devportal.flybits.com/#/documentation/android/push/push-notifications-receiving for more details.");
        return R.mipmap.ic_flybits_notification;
    }

    private String getChannelName(){
        try {
            ComponentName myService = new ComponentName(this, this.getClass());
            Bundle bundle = getPackageManager().getServiceInfo(myService, PackageManager.GET_META_DATA).metaData;

            if (bundle.containsKey("com.flybits.push.notification.title")) {
                return bundle.getString("com.flybits.push.notification.title");
            }
        }catch (PackageManager.NameNotFoundException | NullPointerException e) {
            Logger.exception("FlybitsPushService.getChannelName", e);
        }
        Logger.setTag(_TAG).w("It appears that you have not set your title field within your AndroidManifest declaration of the FlybitsPushService. Please visit https://devportal.flybits.com/#/documentation/android/push/push-notifications-receiving for more details.");
        return "Notifications";
    }
}
