package com.flybits.android.push.analytics

import android.content.Context
import com.flybits.android.push.models.Push
import com.flybits.commons.library.analytics.Analytics

/**
 * This class is responsible for tracking Push analytics. Example of this include:
 * push viewed, push engaged, or push component engaged.
 *
 * @param context The context of the application.
 */
class PushAnalytics(context: Context): Analytics(context) {

    companion object {
        const val analyticsScope = "pushAnalytics"
    }

    private enum class Action(val actionName: String){
        VIEWED("viewed")
        , ENGAGED("engaged")
        , COMPONENT_ENGAGED("componentEngaged")
    }

    private fun track(action: Action, pushId: String, timestamp: Long){
        super.track(Event(analyticsScope, action.actionName, pushId, timestamp))
    }

    /**
     * Track a push item being viewed using a {@link Push} object.
     *
     * @param push {@link Push} being viewed.
     * @param timestamp Timestamp of the event.
     *
     */
    fun trackViewed(push: Push, timestamp: Long = System.currentTimeMillis()){
        track(Action.VIEWED, push.id, timestamp)
    }

    /**
     * Track a push being viewed using the push id.
     *
     * @param pushId The id of the {@link Push} being viewed.
     * @param timestamp Timestamp of the event.
     *
     */
    fun trackViewed(pushId: String, timestamp: Long = System.currentTimeMillis()){
        track(Action.VIEWED, pushId, timestamp)
    }

    /**
     * Track a push being engaged using a {@link Push} object.
     *
     * @param push {@link Push} being engaged.
     * @param timestamp Timestamp of the event.
     *
     */
    fun trackEngaged(push: Push, timestamp: Long = System.currentTimeMillis()){
        track(Action.ENGAGED, push.id, timestamp)
    }

    /**
     * Track a push being engaged using a push id.
     *
     * @param pushId The id of the {@link Push} being engaged.
     * @param timestamp Timestamp of the event.
     *
     */
    fun trackEngaged(pushId: String, timestamp: Long = System.currentTimeMillis()){
        track(Action.ENGAGED, pushId, timestamp)
    }

    /**
     * Track a push component being engaged using a {@link Push} object.
     *
     * @param push {@link Push} associated with the component engaged event.
     * @param timestamp Timestamp of the event.
     *
     */
    fun trackComponentEngaged(push: Push, timestamp: Long = System.currentTimeMillis()){
        track(Action.COMPONENT_ENGAGED, push.id, timestamp)
    }

    /**
     * Track a push component being engaged using an id.
     *
     * @param pushId The id of the {@link Push} being engaged.
     * @param timestamp Timestamp of the event.
     *
     */
    fun trackComponentEngaged(pushId: String, timestamp: Long = System.currentTimeMillis()){
        track(Action.COMPONENT_ENGAGED, pushId, timestamp)
    }
}