package com.flybits.android.push;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.flybits.android.push.api.FlyPushParsing;
import com.flybits.android.push.deserializations.DeserializePushToken;
import com.flybits.android.push.exceptions.FlybitsPushException;
import com.flybits.android.push.models.Push;
import com.flybits.android.push.models.PushToken;
import com.flybits.commons.library.SharedElements;
import com.flybits.commons.library.api.FlyAway;
import com.flybits.commons.library.api.results.BasicResult;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.deserializations.IDeserializer;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.logging.Logger;
import com.flybits.commons.library.models.internal.Result;
import com.google.android.gms.tasks.OnSuccessListener;
import com.google.firebase.iid.FirebaseInstanceId;
import com.google.firebase.iid.InstanceIdResult;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import static com.flybits.android.push.models.PushToken.API;

/**
 * The {@code PushManager} is responsible for all Push related mechanisms such as registering for
 * push notifications, parsing push notifications, and communicating with the FCM servers.
 */
public class PushManager {

    /**
     * Disables Flybits-based push notifications including FCM and foreground notifications. This
     * process will remove the FCM token from the Flybits Push service as well as stopping the
     * foreground Push Service that is internal to the SDK.
     *
     * @param context The context of the activity that is trying to disable FCM push notifications.
     * @param callback The callback used to indicate whether or not disabling push was successful or
     *                 not.
     * @return The network request object that is triggers the {@code callback} based on the network
     * response.
     */
    public static BasicResult disablePush(final Context context, final BasicResultCallback callback){
        return disablePush(context, SharedElements.getSavedJWTToken(context), callback);
    }

    /**
     * Enable Flybits-based push notifications using FCM. This process will retrieve the FCM token
     * from Google and pass it to the Flybits Push Service.
     *
     * @param context The context of the activity that is trying to disable FCM push notifications.
     * @param properties The list of key-value properties that should be added to the Push Token.
     * @param callback The callback used to indicate whether or not disabling push was successful or
     *                 not.
     */
    public static void enableFCMPush(final Context context, @Nullable final HashMap<String, String> properties, final BasicResultCallback callback){
        FirebaseInstanceId.getInstance().getInstanceId().addOnSuccessListener(new OnSuccessListener<InstanceIdResult>() {
            @Override
            public void onSuccess(InstanceIdResult instanceIdResult) {
                final String newToken = instanceIdResult.getToken();
                enablePush(context, newToken, properties, callback);
            }
        });
    }

    /**
     * Enable Flybits-based push notifications. This process will use the {@code token} and pass it
     * to the Flybits Push Service. This {@code token} can be an FCM token or any other one that is
     * necessary to register to push notifications.
     *
     * @param context The context of the activity that is trying to disable FCM push notifications.
     * @param token The push token that should be saved for the logged in user.
     * @param properties The list of key-value properties that should be added to the Push Token.
     * @param callback The callback used to indicate whether or not disabling push was successful or
     *                 not.
     */
    public static void enablePush(final Context context, @NonNull final String token, @Nullable final HashMap<String, String> properties, final BasicResultCallback callback){
        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult resultObject = new BasicResult(context, callback, executorService, handler);
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    PushToken resultSerializationObj    = new PushToken(token, properties);
                    DeserializePushToken deserialization = new DeserializePushToken();
                    final String json = deserialization.toJson(resultSerializationObj);
                    final Result pushResult = FlyAway.post(context, API, json, null, "PushManager.enablePush", null);
                    resultObject.setResult(pushResult);
                }catch (final FlybitsException e){
                    resultObject.setFailed(e);
                }
            }
        });
    }

    /**
     * Parse a FCM {@link Push} notification that is received from the Flybits platform. This push
     * notification is provided in a specific Flybits format which the SDK can parse. If the push
     * notification is not a Flybits notification and therefore cannot be parsed a
     * {@link FlybitsPushException} will be thrown within the
     * {@code PendingRequest.RequestCallbackWithResponse#onException(FlybitsException)}.
     *
     * @param mContext The {@link Context} of the application.
     * @param map The {@link Map} which contains the properties sent from the Flybits
     *               platform. This should not be tempered with and should be sent directly to this
     *               method from the FCM receiver.
     * @param callback The callback that initiated when the request is completed. It will contain
     *                 either a successful method or failure with a
     *                 {@code FlybitsException} which
     *                 indicates the reason for failure.
     * @param <T> The {@code body} of the custom fields option. Some push notification may
     *            provide additional fields with the request in order provide mechanisms for
     *            deep-linking hence a unique structure can be provided.
     */
    public static <T extends IDeserializer> void parsePushNotification(@NonNull Context mContext, @NonNull Map map, ObjectResultCallback<Push> callback){
        try {
            Push push = FlyPushParsing.parsePushNotification(mContext, map);
            if (callback != null) {
                callback.onSuccess(push);
            }
        }catch(FlybitsPushException e){
            if (callback != null) {
                callback.onException(e);
            }
        }
    }

    static BasicResult disablePush(final Context context, @NonNull final String jwttoken, final BasicResultCallback callback){

        if (jwttoken == null){
            BasicResult result = new BasicResult(context, callback);
            result.setFailed(new FlybitsException("Your JWT Token appears to be null. Please make sure you have set to disable push."));
            return result;
        }

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult resultObject    = new BasicResult(context, callback, executorService, handler);
        executorService.execute(new Runnable() {
            @Override
            public void run() {
                try {
                    HashMap<String, String> mapOfHeaders    = new HashMap<String, String>();
                    mapOfHeaders.put("X-Authorization", jwttoken);
                    final Result result = FlyAway.delete(context, API, mapOfHeaders, "PushManager.disablePush", null);
                    resultObject.setResult(result);
                }catch (final FlybitsException e){
                    Logger.exception("PushManager.disablePush", e);
                    resultObject.setFailed(e);
                }
            }
        });
        return resultObject;
    }
}
