package com.flybits.android.push;

import android.annotation.TargetApi;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.VectorDrawable;
import android.os.AsyncTask;
import android.os.Build;
import android.support.annotation.ColorInt;
import android.support.annotation.DrawableRes;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v4.app.NotificationCompat;
import android.support.v4.content.ContextCompat;
import android.text.TextUtils;

import com.flybits.android.push.models.Push;
import com.flybits.android.push.models.pushTypes.ContentMetadata;
import com.flybits.android.push.models.pushTypes.WeblinkMetadata;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;

/**
 * The purpose of this class is simplify the process of displaying a push notification in the
 * notification tray of the device. It reduces the complexities figuring out which feature are
 * available with the different versions of the Android OS. It also eliminates redundant code for
 * the Notification and NotificationChannel(Available for version 8.0+) classes such as vibration
 * and lights. This class also provides an option to retrieve the {@code NotificationCompat.Builder}
 * and {@code NotificationChannel} objects so that you can add additional options that might not be
 * implemented in the future that this class currently does not support.
 */
public class FlybitsNotificationManager {

    private NotificationChannel notificationChannel;
    private NotificationCompat.Builder notification;
    private Context context;
    private String id;

    private FlybitsNotificationManager(Simplifier builder){

        if (builder.notification != null){
            notification        = builder.notification;
        }

        if (builder.context != null){
            context             = builder.context;
        }
        if (builder.id != null){
            id             = builder.id;
        }

        if (builder.notificationChannel != null){
            notificationChannel = builder.notificationChannel;
        }
    }

    /**
     * Get the {@code NotificationCompat.Builder} object that the application can add its additional
     * methods that might not be implemented by the {@link Simplifier}.
     *
     * @return The {@code NotificationCompat.Builder} object that can be converted to a Notification
     * to be displayed to the end user.
     */
    public NotificationCompat.Builder get(){
        return notification;
    }

    /**
     * Get the {@code NotificationChannel} that is used to set the Notification Channel of the
     * notification. The {@code NotificationChannel} is only supported on Android OS 8.0+.
     *
     * @return The {@code NotificationChannel} that is attached to the Notification, however this is
     * only supported on Android OS 8.0+ devices.
     */
    public NotificationChannel getChannel(){
        return notificationChannel;
    }

    /**
     * Displays the notification within the Notification tray of the device. This method handles any
     * additional code that is needed for the various Android OS versions so that it does not need
     * to be implemented/handled by the application, this includes setting the
     * {@code NotificationChannel} for Android OS 8.0+ devices.
     */
    public void show(){
        NotificationManager mNotificationManager = (NotificationManager)
                context.getSystemService(Context.NOTIFICATION_SERVICE);

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            if (mNotificationManager != null && notificationChannel != null){
                mNotificationManager.createNotificationChannel(notificationChannel);
            }
        }

        if (mNotificationManager != null){
            int hashOfID    = id.hashCode();
            mNotificationManager.notify(hashOfID, notification.build());
        }
    }

    /**
     * The {@code Simplifier} class is responsible for building the
     * {@link FlybitsNotificationManager} object that is used to display the Notification. This
     * class handles any complexities in order to simplify the process of building your
     * Notification.
     */
    public static class Simplifier {

        private NotificationChannel notificationChannel;
        private NotificationCompat.Builder notification;
        private Context context;
        private String id;

        private String channelName;
        private String channelId;
        private int importance;

        private String title;
        private String body;

        private Bitmap bitmap;

        private NotificationStyle style;

        private long[] vibration;
        private int argbLights;

        /**
         * Constructor that defines all the necessary components needed to build a Notification. The
         * {@code title} and {@code body} are nullable but all other parameters are mandatory.
         *
         * @param context The context of the application.
         * @param id The unique identifier used to define the Notification.
         * @param title The title to be display as part of the Notification. Null can be used for
         *              this parameter.
         * @param body The body/message that is part of the Notification to be displayed. Null can
         *             be used for this parameter.
         * @param resourceSmallIcon The resource that indicates which small icon should be displayed.
         */
        public Simplifier(@NonNull Context context, @NonNull String id, @Nullable String title,
                          @Nullable String body, @DrawableRes int resourceSmallIcon){

            this.id             = id;
            this.context        = context;
            this.channelName    = "channelFlybitsSDK";
            this.channelId      = "channelFlybitsSDK_ID";
            this.title          = title;
            this.body           = body;
            this.importance     = 3; //3 is NotificationManager.IMPORTANCE_DEFAULT on used for OS 8.0+
            this.style          = NotificationStyle.BIG_TEXT;
            notification        = new NotificationCompat.Builder(context, channelId).setAutoCancel(true);

            if (title != null) {
                notification.setContentTitle(title);
            }

            if (body != null) {
                notification.setContentText(body);
            }

            notification.setSmallIcon(resourceSmallIcon);
            setBadgeIcon(NotificationCompat.BADGE_ICON_SMALL);
        }

        /**
         * Constructor that defines all the necessary components needed to build a Notification
         * using the Flybits {@link Push} object.
         *
         * @param context The context of the application.
         * @param push The parsed Flybits {@link Push} that should be displayed.
         * @param resourceSmallIcon The resource that indicates which small icon should be displayed.
         */
        public Simplifier(@NonNull Context context, @NonNull Push push, @DrawableRes int resourceSmallIcon){
            this(context, push.getId(), push.getTitle(), push.getMessage(), resourceSmallIcon);

            if (push.getCustomFieldsAsString() != null && push.getMetadataID() != null
                    && !TextUtils.isEmpty(push.getCustomFieldsAsString())){

                switch (push.getMetadataID()){
                    case BuildConfig.METADATA_ID_CONTENT:
                        ContentMetadata metadata = new ContentMetadata(push.getCustomFieldsAsString());
                        if (metadata.getContentId() != null && metadata.onClick(context) != null){
                            setPendingIntent(metadata.onClick(context));
                        }
                        break;
                    case BuildConfig.METADATA_ID_WEB:
                        WeblinkMetadata metadata1 = new WeblinkMetadata(push.getCustomFieldsAsString());
                        if (metadata1.getUrl() != null && metadata1.onClick(context) != null){
                            setPendingIntent(metadata1.onClick(context));
                        }
                        break;
                }
            }
        }

        /**
         * Adds a list of {@code NotificationCompat.Action}s that should be displayed to the
         * end-user as part of the Notification.
         *
         * @param actions The list of {@code NotificationCompat.Action}s that can be added to the
         *                notification.
         *
         * @return The {@code Simplifier} that can be passed to the
         * {@link FlybitsNotificationManager} class in order to construct the appropriate
         * Notification.
         */
        public Simplifier addActions(NotificationCompat.Action... actions){
            if (actions != null && actions.length > 0){
                for (NotificationCompat.Action action : actions){
                    notification.addAction(action);
                }
            }
            return this;
        }

        /**
         * Sets which icon to display as a badge for this notification.
         *
         * @param iconType Must be one of {@code NotificationCompat.BADGE_ICON_NONE},
         * {@code NotificationCompat.BADGE_ICON_SMALL}, {@code NotificationCompat.BADGE_ICON_LARGE}
         *
         * @return The {@code Simplifier} that can be passed to the
         * {@link FlybitsNotificationManager} class in order to construct the appropriate
         * Notification.
         */
        public Simplifier setBadgeIcon(@NotificationCompat.BadgeIconType int iconType){
            notification.setBadgeIconType(iconType);
            return this;
        }

        /**
         * Create a Notification channel.
         *
         * @param channelId The id of the channel. Must be unique per package. The value may be
         *                  truncated if it is too long.
         * @param channelName The user visible name of the channel. You can rename this channel when
         *                    the system locale changes by listening for the
         *                    {@code Intent.ACTION_LOCALE_CHANGED} broadcast. The recommended
         *                    maximum length is 40 characters; the value may be truncated if it is
         *                    too long.
         * @param importance The importance of the channel. This controls how interruptive
         *                   notifications posted to this channel are.
         * @return The {@code Simplifier} that can be passed to the
         * {@link FlybitsNotificationManager} class in order to construct the appropriate
         * Notification.
         */
        public Simplifier setChannel(@NonNull String channelId, @NonNull String channelName, int importance){

            this.channelName    = channelName;
            this.channelId      = channelId;
            this.importance     = importance;

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                notification.setChannelId(channelId);
            }
            return this;
        }

        /**
         * Set the large icon that is shown in the ticker and notification through a Bitmap.
         *
         * @param icon The Bitmap to be displayed.
         * @param displayImageInBigStyle true indicates that the Icon should be displayed in a Big
         *                               Style notification.
         *
         * @return The {@code Simplifier} that can be passed to the
         * {@link FlybitsNotificationManager} class in order to construct the appropriate
         * Notification.
         */
        public Simplifier setIcon(Bitmap icon, boolean displayImageInBigStyle){

            if (icon != null) {
                notification.setLargeIcon(icon);
                this.bitmap = icon;

                if (displayImageInBigStyle) {
                    this.style = NotificationStyle.BIG_PICTURE;
                }
            }

            return this;
        }

        /**
         * Set the large icon that is shown in the ticker and notification through an application
         * resource.
         *
         * @param resourceIcon The resource used to construct the Icon to be displayed.
         * @param displayImageInBigStyle true indicates that the Icon should be displayed in a Big
         *                               Style notification.
         *
         * @return The {@code Simplifier} that can be passed to the
         * {@link FlybitsNotificationManager} class in order to construct the appropriate
         * Notification.
         */
        public Simplifier setIcon(int resourceIcon, boolean displayImageInBigStyle){
            Bitmap bitmap = getBitmap(context, resourceIcon);
            setIcon(bitmap, displayImageInBigStyle);
            return this;
        }

        /**
         * Set the argb value that you would like the LED on the device to blink, as well as the
         * rate. The rate is specified in terms of the number of milliseconds to be on
         * and then the number of milliseconds to be off.
         *
         * @param argb Value that you would like the LED on the device to blink.
         * @param onMs Milliseconds rate that the lights should be on.
         * @param offMs Milliseconds rate that the lights should be off.
         *
         * @return The {@code Simplifier} that can be passed to the
         * {@link FlybitsNotificationManager} class in order to construct the appropriate
         * Notification.
         */
        public Simplifier setLights(@ColorInt int argb, int onMs, int offMs){
            this.argbLights = argb;
            notification.setLights(argb, onMs, offMs);
            return this;
        }

        /**
         * Supply a {@code PendingIntent} to send when the notification is clicked.
         *
         * @param intent The {@code PendingIntent} that should send when the notification is
         *               clicked.
         *
         * @return The {@code Simplifier} that can be passed to the
         * {@link FlybitsNotificationManager} class in order to construct the appropriate
         * Notification.
         */
        public Simplifier setPendingIntent(@NonNull PendingIntent intent){
            notification.setContentIntent(intent);
            return this;
        }

        /**
         * Set the vibration pattern to use.
         *
         * @param vibration The pattern to use for vibration once the notification is displayed.
         *
         * @return The {@code Simplifier} that can be passed to the
         * {@link FlybitsNotificationManager} class in order to construct the appropriate
         * Notification.
         */
        public Simplifier setVibration(long[] vibration){
            if (vibration != null && vibration.length > 0) {
                this.vibration = vibration;
                notification.setVibrate(vibration);
            }
            return this;
        }

        /**
         * Create the {@link FlybitsNotificationManager} based on all the options that have been
         * set.
         *
         * @return The {@link FlybitsNotificationManager} object that is used to display the
         * Notification.
         */
        public FlybitsNotificationManager build(){

            //As part of Android 8 you need to have a channel to properly display the notification
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                notificationChannel = new NotificationChannel(
                        channelId, channelName, importance);

                if (vibration != null){
                    notificationChannel.enableVibration(true);
                    notificationChannel.setVibrationPattern(vibration);
                }

                if (argbLights != 0){
                    notificationChannel.enableLights(true);
                    notificationChannel.setLightColor(argbLights);
                }
            }

            switch (style){
                case BIG_TEXT:
                    NotificationCompat.BigTextStyle bigText = new NotificationCompat.BigTextStyle();
                    bigText.setBigContentTitle(title);
                    bigText.setSummaryText(body);
                    notification.setStyle(bigText);
                    break;
                case BIG_PICTURE:
                    NotificationCompat.BigPictureStyle bigPicture = new NotificationCompat.BigPictureStyle();
                    bigPicture.bigPicture(bitmap).bigLargeIcon(null);
                    bigPicture.setBigContentTitle(title);
                    notification.setStyle(bigPicture);
                    break;
            }

            return new FlybitsNotificationManager(this);
        }
    }

    private enum NotificationStyle{

        BIG_TEXT,

        BIG_PICTURE
    }

    private static Bitmap getBitmap(Context context, int drawableId) {
        Drawable drawable = ContextCompat.getDrawable(context, drawableId);
        if (drawable instanceof BitmapDrawable) {
            return BitmapFactory.decodeResource(context.getResources(), drawableId);
        } else if (drawable instanceof VectorDrawable) {
            return getBitmap((VectorDrawable) drawable);
        }
        return null;
    }

    @TargetApi(Build.VERSION_CODES.LOLLIPOP)
    private static Bitmap getBitmap(VectorDrawable vectorDrawable) {
        Bitmap bitmap = Bitmap.createBitmap(vectorDrawable.getIntrinsicWidth(),
                vectorDrawable.getIntrinsicHeight(), Bitmap.Config.ARGB_8888);
        Canvas canvas = new Canvas(bitmap);
        vectorDrawable.setBounds(0, 0, canvas.getWidth(), canvas.getHeight());
        vectorDrawable.draw(canvas);
        return bitmap;
    }
}
