package com.flybits.android.push.models

/**
 * The Entity that the Push notification directly affects.
 *
 * @param key the [String] value representing each [PushEntity] option.
 */
enum class PushEntity(val key: String) {

    /**
     * Indicates that a push notification was delivered with a custom body response.
     */
    CUSTOM("custom"),

    /**
     * Indicates that the push notification directly affects a Context `Rule` entity.
     */
    RULE("rule"),

    /**
     * Indicates that the push notification directly affects an entity.
     */
    EXPERIENCE("experience"),

    /**
     * Indicates that the push notification directly affects a Group entity.
     */
    GROUP("group"),

    /**
     * Indicates that the push notification directly affects a Content entity.
     */
    CONTENT("content"),

    /**
     * Indicates that the push notification directly affects a Content Data entity.
     */
    CONTENT_DATA("contentData"),

    /**
     * Indicates that a unknown push notification was received from the Flybits Push Server.
     */
    UNKNOWN("N/A");


    companion object {

        /**
         * Get the [PushEntity] enum value corresponding to a [String] representation.
         *
         * @param key the [String] representation of the [PushEntity] enum.
         *
         * @return The [PushEntity] enum for the [String] representation.
         */
        @JvmStatic
        fun fromKey(key: String?): PushEntity {
            if (key == null) {
                return UNKNOWN
            }

            for (type in values()) {
                if (type.key.equals(key, ignoreCase = true)) {
                    return type
                }
            }
            return UNKNOWN
        }
    }
}

