package com.flybits.android.push.models

/**
 * The [PushCategory] indicates which entity category the push notification effects. Below is the
 * the complete list of entity categories.
 *
 *  [PushCategory.CONTEXT]
 *  [PushCategory.KERNEL]
 *  [PushCategory.CUSTOM]
 *
 * @param key the String value representing each [PushCategory] option.
 *
 */
enum class PushCategory(val key: String) {

    /**
     * Indicates that the entity associated to this notification belongs to the `CONTEXT`
     * category.
     */
    CONTEXT("context"),

    /**
     * Indicates that the entity associated to this notification belongs to the `CUSTOM`
     * category.
     */
    CUSTOM("custom"),

    /**
     * Indicates that the entity associated to this notification belongs to the `KERNEL`
     * category.
     */
    KERNEL("kernel"),

    /**
     * Indicates that a unknown category was received with the specified entity.
     */
    UNKNOWN("");

    companion object {

        /**
         * Get the `PushCategory` enum value corresponding to an String representation.
         *
         * @param key the String representation of the `PushCategory` enum.
         *
         * @return The `PushCategory` enum for the String representation.
         */
        @JvmStatic
        fun fromKey(key: String?): PushCategory {
            if (key == null) {
                return UNKNOWN
            }

            for (type in values()) {
                if (type.key.equals(key, ignoreCase = true)) {
                    return type
                }
            }
            return UNKNOWN
        }
    }
}